<?php
/*
Plugin Name: Simple Raffle Drawings
Description: A simple raffle drawing plugin that allows users to enter their details for a raffle drawing. Each drawing is embeddable via shortcode and only one entry per person (based on email) is allowed. The admin dashboard includes pages to manage raffles (with a GUI editor for custom entry fields), view results, edit, delete, reset raffles, plus a settings page for customization. When a winner is auto-selected, the raffle is closed and only the winner's name is shown on the front end. An admin option to remove the winner (and reopen the raffle) is also provided.
Version: 1.5.0
Author: FrostLine Works
Author URI: https://frostlineworks.com
Tested up to: 6.2
Requires Plugins: flwsecureupdates
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}


class Simple_Raffle_Plugin {

    private $drawings_table;
    private $entries_table;

    public function __construct() {
        global $wpdb;
        $this->drawings_table = $wpdb->prefix . 'raffle_drawings';
        $this->entries_table  = $wpdb->prefix . 'raffle_entries';

        // Activation hook creates/updates tables.
        register_activation_hook( __FILE__, array( $this, 'activate_plugin' ) );
        // Deactivation hook checks settings to remove tables if toggled.
        register_deactivation_hook( __FILE__, array( 'Simple_Raffle_Plugin', 'deactivate_plugin' ) );

        // Register admin menus and settings.
        add_action( 'admin_menu', array( $this, 'register_admin_menu' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );

        // Shortcode and front-end submission handler.
        add_shortcode( 'raffle', array( $this, 'raffle_shortcode' ) );
        add_action( 'init', array( $this, 'handle_entry_submission' ) );
    }

    /**
     * Create or update the custom tables for raffles and entries.
     */
    public function activate_plugin() {
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();

        $sql_drawings = "CREATE TABLE {$this->drawings_table} (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            title varchar(255) NOT NULL,
            description text,
            form_fields text,
            winner_entry_id int(11) DEFAULT NULL,
            PRIMARY KEY  (id)
        ) $charset_collate;";

        $sql_entries = "CREATE TABLE {$this->entries_table} (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            raffle_id mediumint(9) NOT NULL,
            name varchar(255) NOT NULL,
            phone varchar(50),
            shipping_address text NOT NULL,
            email varchar(255) NOT NULL,
            custom_data text,
            entry_date datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY  (id),
            UNIQUE KEY unique_entry (raffle_id, email)
        ) $charset_collate;";

        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
        dbDelta( $sql_drawings );
        dbDelta( $sql_entries );
    }

    /**
     * Remove database tables on deactivation if the setting is toggled.
     */
    public static function deactivate_plugin() {
        global $wpdb;
        $options = get_option( 'simple_raffle_options', array() );
        if ( ! empty( $options['remove_tables_on_deactivation'] ) ) {
            $drawings_table = $wpdb->prefix . 'raffle_drawings';
            $entries_table  = $wpdb->prefix . 'raffle_entries';
            $wpdb->query( "DROP TABLE IF EXISTS $drawings_table" );
            $wpdb->query( "DROP TABLE IF EXISTS $entries_table" );
        }
    }

    /**
     * Register admin menu pages.
     */
    public function register_admin_menu() {
        add_menu_page(
            'Raffles',
            'Raffles',
            'manage_options',
            'simple-raffles',
            array( $this, 'admin_raffle_list_page' ),
            'dashicons-awards'
        );

        add_submenu_page(
            'simple-raffles',
            'Add New Raffle Drawing',
            'Add New',
            'manage_options',
            'simple-raffles-add',
            array( $this, 'admin_add_raffle_page' )
        );

        add_submenu_page(
            'simple-raffles',
            'Raffle Settings',
            'Settings',
            'manage_options',
            'simple-raffle-settings',
            array( $this, 'admin_settings_page' )
        );

        // Hidden pages.
        add_submenu_page(
            null,
            'View Raffle Results',
            'View Raffle Results',
            'manage_options',
            'simple-raffles-view',
            array( $this, 'admin_view_entries_page' )
        );
        add_submenu_page(
            null,
            'Edit Raffle',
            'Edit Raffle',
            'manage_options',
            'simple-raffles-edit',
            array( $this, 'admin_edit_raffle_page' )
        );
        add_submenu_page(
            null,
            'Delete Raffle',
            'Delete Raffle',
            'manage_options',
            'simple-raffles-delete',
            array( $this, 'admin_delete_raffle_page' )
        );
        add_submenu_page(
            null,
            'Reset Raffle',
            'Reset Raffle',
            'manage_options',
            'simple-raffles-reset',
            array( $this, 'admin_reset_raffle_page' )
        );
        add_submenu_page(
            null,
            'Remove Winner',
            'Remove Winner',
            'manage_options',
            'simple-raffles-remove-winner',
            array( $this, 'admin_remove_winner_page' )
        );
        add_submenu_page(
            null,
            'Edit Entry',
            'Edit Entry',
            'manage_options',
            'simple-raffles-edit-entry',
            array( $this, 'admin_edit_entry_page' )
        );
    }

    /**
     * Register settings for the Raffle plugin.
     */
    public function register_settings() {
        register_setting( 'simple_raffle_options_group', 'simple_raffle_options' );
    }

    // ---------------------
    // Admin: Raffle Settings Page
    // ---------------------
    public function admin_settings_page() {
        $options = get_option( 'simple_raffle_options' );
        $form_title_font_size = isset($options['form_title_font_size']) ? $options['form_title_font_size'] : '24px';
        $form_title_color = isset($options['form_title_color']) ? $options['form_title_color'] : '#333333';
        $field_font_size = isset($options['field_font_size']) ? $options['field_font_size'] : '16px';
        $field_color = isset($options['field_color']) ? $options['field_color'] : '#333333';
        $form_background_color = isset($options['form_background_color']) ? $options['form_background_color'] : '#f9f9f9';
        $submit_button_text = isset($options['submit_button_text']) ? $options['submit_button_text'] : 'Enter Raffle';
        $submit_button_text_color = isset($options['submit_button_text_color']) ? $options['submit_button_text_color'] : '#ffffff';
        $submit_button_background_color = isset($options['submit_button_background_color']) ? $options['submit_button_background_color'] : '#0073aa';
        $submit_button_font_size = isset($options['submit_button_font_size']) ? $options['submit_button_font_size'] : '16px';

        $winner_display_text = isset($options['winner_display_text']) ? $options['winner_display_text'] : 'Winner:';
        $winner_display_font_size = isset($options['winner_display_font_size']) ? $options['winner_display_font_size'] : '20px';
        $winner_display_color = isset($options['winner_display_color']) ? $options['winner_display_color'] : '#000000';

        $remove_tables = isset($options['remove_tables_on_deactivation']) ? $options['remove_tables_on_deactivation'] : '';

        ?>
        <div class="wrap">
            <h1>Raffle Settings</h1>
            <form method="post" action="options.php">
                <?php settings_fields( 'simple_raffle_options_group' ); ?>
                <table class="form-table">
                    <tr valign="top">
                        <th scope="row">Form Title Font Size</th>
                        <td><input type="text" name="simple_raffle_options[form_title_font_size]" value="<?php echo esc_attr( $form_title_font_size ); ?>"></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Form Title Color</th>
                        <td><input type="color" name="simple_raffle_options[form_title_color]" value="<?php echo esc_attr( $form_title_color ); ?>"></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Field Font Size</th>
                        <td><input type="text" name="simple_raffle_options[field_font_size]" value="<?php echo esc_attr( $field_font_size ); ?>"></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Field Text Color</th>
                        <td><input type="color" name="simple_raffle_options[field_color]" value="<?php echo esc_attr( $field_color ); ?>"></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Form Background Color</th>
                        <td><input type="color" name="simple_raffle_options[form_background_color]" value="<?php echo esc_attr( $form_background_color ); ?>"></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Submit Button Text</th>
                        <td><input type="text" name="simple_raffle_options[submit_button_text]" value="<?php echo esc_attr( $submit_button_text ); ?>"></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Submit Button Text Color</th>
                        <td><input type="color" name="simple_raffle_options[submit_button_text_color]" value="<?php echo esc_attr( $submit_button_text_color ); ?>"></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Submit Button Background Color</th>
                        <td><input type="color" name="simple_raffle_options[submit_button_background_color]" value="<?php echo esc_attr( $submit_button_background_color ); ?>"></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Submit Button Font Size</th>
                        <td><input type="text" name="simple_raffle_options[submit_button_font_size]" value="<?php echo esc_attr( $submit_button_font_size ); ?>"></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Winner Display Text</th>
                        <td><input type="text" name="simple_raffle_options[winner_display_text]" value="<?php echo esc_attr( $winner_display_text ); ?>"></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Winner Display Font Size</th>
                        <td><input type="text" name="simple_raffle_options[winner_display_font_size]" value="<?php echo esc_attr( $winner_display_font_size ); ?>"></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Winner Display Color</th>
                        <td><input type="color" name="simple_raffle_options[winner_display_color]" value="<?php echo esc_attr( $winner_display_color ); ?>"></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Remove Database Tables on Deactivation</th>
                        <td>
                            <input type="checkbox" name="simple_raffle_options[remove_tables_on_deactivation]" value="1" <?php checked( $remove_tables, 1 ); ?> />
                            <span class="description">If checked, the plugin will drop its custom tables when deactivated.</span>
                        </td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
            <a href="<?php echo admin_url('admin.php?page=simple-raffles'); ?>">Back to Raffles</a>
        </div>
        <?php
    }

    // ---------------------
    // Admin: List All Raffles
    // ---------------------
    public function admin_raffle_list_page() {
        global $wpdb;
        $drawings = $wpdb->get_results("SELECT * FROM {$this->drawings_table}");
        ?>
        <div class="wrap">
            <h1>Raffle Drawings</h1>
            <?php if (isset($_GET['raffle_updated'])): ?>
                <div class="notice notice-success is-dismissible"><p>Raffle updated successfully.</p></div>
            <?php endif; ?>
            <?php if (isset($_GET['raffle_deleted'])): ?>
                <div class="notice notice-success is-dismissible"><p>Raffle deleted successfully.</p></div>
            <?php endif; ?>
            <?php if (isset($_GET['raffle_reset'])): ?>
                <div class="notice notice-success is-dismissible"><p>Raffle entries reset successfully.</p></div>
            <?php endif; ?>
            <?php if (isset($_GET['winner_removed'])): ?>
                <div class="notice notice-success is-dismissible"><p>Winner removed and raffle reopened.</p></div>
            <?php endif; ?>

            <a href="<?php echo admin_url('admin.php?page=simple-raffles-add'); ?>" class="page-title-action">Add New Raffle</a>
            <table class="wp-list-table widefat fixed striped" style="margin-top:20px;">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Title</th>
                        <th>Shortcode</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($drawings): ?>
                        <?php foreach ($drawings as $drawing): ?>
                            <?php $shortcode = '[raffle id="' . esc_attr($drawing->id) . '"]'; ?>
                            <tr>
                                <td><?php echo esc_html($drawing->id); ?></td>
                                <td><?php echo esc_html($drawing->title); ?></td>
                                <td><input type="text" value="<?php echo esc_attr($shortcode); ?>" readonly style="width:200px;"></td>
                                <td>
                                    <a href="<?php echo admin_url('admin.php?page=simple-raffles-view&id=' . $drawing->id); ?>">View Results</a> |
                                    <a href="<?php echo admin_url('admin.php?page=simple-raffles-edit&id=' . $drawing->id); ?>">Edit</a> |
                                    <a href="<?php echo admin_url('admin.php?page=simple-raffles-delete&id=' . $drawing->id . '&raffle_nonce=' . wp_create_nonce('delete_raffle_' . $drawing->id)); ?>">Delete</a> |
                                    <a href="<?php echo admin_url('admin.php?page=simple-raffles-reset&id=' . $drawing->id . '&raffle_nonce=' . wp_create_nonce('reset_raffle_' . $drawing->id)); ?>">Reset</a>
                                    <?php if (!empty($drawing->winner_entry_id)): ?>
                                        | <a href="<?php echo admin_url('admin.php?page=simple-raffles-remove-winner&id=' . $drawing->id . '&raffle_nonce=' . wp_create_nonce('remove_winner_' . $drawing->id)); ?>">Remove Winner</a>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <tr><td colspan="4">No raffle drawings found.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    // ---------------------
    // Admin: Add Raffle Page (with GUI Editor)
    // ---------------------
    public function admin_add_raffle_page() {
        if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['raffle_nonce']) && wp_verify_nonce($_POST['raffle_nonce'], 'add_raffle')) {
            $title = isset($_POST['title']) ? sanitize_text_field(wp_unslash($_POST['title'])) : '';
            $description = isset($_POST['description']) ? sanitize_textarea_field(wp_unslash($_POST['description'])) : '';

            $fields = isset($_POST['fields']) ? $_POST['fields'] : array();
            $clean_fields = array();
            if (!empty($fields) && is_array($fields)) {
                foreach ($fields as $field) {
                    $clean_fields[] = array(
                        'label'       => isset($field['label']) ? sanitize_text_field($field['label']) : '',
                        'name'        => isset($field['name']) ? sanitize_text_field($field['name']) : '',
                        'type'        => isset($field['type']) ? sanitize_text_field($field['type']) : 'text',
                        'required'    => isset($field['required']) ? true : false,
                        'placeholder' => isset($field['placeholder']) ? sanitize_text_field($field['placeholder']) : '',
                        'options'     => isset($field['options']) ? array_map('sanitize_text_field', explode("\n", trim($field['options']))) : array(),
                    );
                }
            }
            $form_fields = !empty($clean_fields) ? wp_json_encode($clean_fields) : '';

            if (!empty($title)) {
                global $wpdb;
                $wpdb->insert(
                    $this->drawings_table,
                    array(
                        'title'       => $title,
                        'description' => $description,
                        'form_fields' => $form_fields,
                    )
                );
                wp_redirect(admin_url('admin.php?page=simple-raffles'));
                exit;
            } else {
                echo '<div class="notice notice-error"><p>Title is required.</p></div>';
            }
        }
        ?>
        <div class="wrap">
            <h1>Add New Raffle Drawing</h1>
            <form method="post">
                <?php wp_nonce_field('add_raffle', 'raffle_nonce'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="title">Title <span style="color:red">*</span></label></th>
                        <td><input type="text" name="title" id="title" class="regular-text" required></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="description">Description</label></th>
                        <td><textarea name="description" id="description" class="large-text" rows="5"></textarea></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="custom_fields">Custom Entry Fields</label></th>
                        <td>
                            <div id="custom-fields-container">
                                <div class="custom-field-block">
                                    <h3>Field <span class="cf-number">1</span></h3>
                                    <p>
                                        <label>Field Label:</label><br>
                                        <input type="text" name="fields[0][label]" class="regular-text" required>
                                    </p>
                                    <p>
                                        <label>Field Name:</label><br>
                                        <input type="text" name="fields[0][name]" class="regular-text" required>
                                    </p>
                                    <p>
                                        <label>Field Type:</label><br>
                                        <select name="fields[0][type]" class="field-type">
                                            <option value="text">Text</option>
                                            <option value="textarea">Textarea</option>
                                            <option value="select">Select</option>
                                        </select>
                                    </p>
                                    <p>
                                        <label>Required:</label>
                                        <input type="checkbox" name="fields[0][required]" value="1">
                                    </p>
                                    <p>
                                        <label>Placeholder:</label><br>
                                        <input type="text" name="fields[0][placeholder]" class="regular-text">
                                    </p>
                                    <div class="field-options" style="display:none;">
                                        <p>
                                            <label>Options (one per line):</label><br>
                                            <textarea name="fields[0][options]" rows="3" class="large-text"></textarea>
                                        </p>
                                    </div>
                                </div>
                            </div>
                            <p>
                                <button type="button" id="add-field" class="button">Add Field</button>
                            </p>
                            <script>
                                document.addEventListener("DOMContentLoaded", function() {
                                    let fieldIndex = 1;
                                    document.getElementById("add-field").addEventListener("click", function() {
                                        const container = document.getElementById("custom-fields-container");
                                        const block = document.createElement("div");
                                        block.className = "custom-field-block";
                                        block.innerHTML = '<h3>Field <span class="cf-number">' + (fieldIndex + 1) + '</span></h3>' +
                                            '<p><label>Field Label:</label><br>' +
                                            '<input type="text" name="fields[' + fieldIndex + '][label]" class="regular-text" required></p>' +
                                            '<p><label>Field Name:</label><br>' +
                                            '<input type="text" name="fields[' + fieldIndex + '][name]" class="regular-text" required></p>' +
                                            '<p><label>Field Type:</label><br>' +
                                            '<select name="fields[' + fieldIndex + '][type]" class="field-type">' +
                                                '<option value="text">Text</option>' +
                                                '<option value="textarea">Textarea</option>' +
                                                '<option value="select">Select</option>' +
                                            '</select></p>' +
                                            '<p><label>Required:</label> ' +
                                            '<input type="checkbox" name="fields[' + fieldIndex + '][required]" value="1"></p>' +
                                            '<p><label>Placeholder:</label><br>' +
                                            '<input type="text" name="fields[' + fieldIndex + '][placeholder]" class="regular-text"></p>' +
                                            '<div class="field-options" style="display:none;">' +
                                                '<p><label>Options (one per line):</label><br>' +
                                                '<textarea name="fields[' + fieldIndex + '][options]" rows="3" class="large-text"></textarea></p>' +
                                            '</div>';
                                        container.appendChild(block);
                                        fieldIndex++;
                                        addFieldTypeListener(block);
                                    });
                                    function addFieldTypeListener(block) {
                                        const select = block.querySelector(".field-type");
                                        select.addEventListener("change", function() {
                                            const optionsDiv = block.querySelector(".field-options");
                                            optionsDiv.style.display = (select.value === "select") ? "block" : "none";
                                        });
                                    }
                                    document.querySelectorAll(".custom-field-block").forEach(function(block) {
                                        addFieldTypeListener(block);
                                    });
                                });
                            </script>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Add Raffle Drawing'); ?>
            </form>
            <a href="<?php echo admin_url('admin.php?page=simple-raffles'); ?>">Back to Raffle Drawings</a>
        </div>
        <?php
    }

    // ---------------------
    // Admin: Edit Entry Page (Hidden)
    // ---------------------
    public function admin_edit_entry_page() {
        global $wpdb;
        if ( ! isset( $_GET['entry_id'] ) ) {
            echo '<div class="wrap"><h1>No entry specified.</h1></div>';
            return;
        }
        $entry_id = intval( $_GET['entry_id'] );
        $entry = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->entries_table} WHERE id = %d", $entry_id ) );
        if ( ! $entry ) {
            echo '<div class="wrap"><h1>Entry not found.</h1></div>';
            return;
        }
        $raffle = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->drawings_table} WHERE id = %d", $entry->raffle_id ) );
        if ( ! $raffle ) {
            echo '<div class="wrap"><h1>Associated raffle not found.</h1></div>';
            return;
        }
        $nonce = isset( $_GET['entry_nonce'] ) ? $_GET['entry_nonce'] : '';
        if ( ! wp_verify_nonce( $nonce, 'edit_entry_' . $entry_id ) ) {
            echo '<div class="wrap"><h1>Invalid nonce.</h1></div>';
            return;
        }
        if ( $_SERVER['REQUEST_METHOD'] === 'POST' && isset( $_POST['edit_entry_nonce'] ) && wp_verify_nonce( $_POST['edit_entry_nonce'], 'edit_entry_' . $entry_id ) ) {
            $name = isset( $_POST['name'] ) ? sanitize_text_field( wp_unslash( $_POST['name'] ) ) : '';
            $phone = isset( $_POST['phone'] ) ? sanitize_text_field( wp_unslash( $_POST['phone'] ) ) : '';
            $shipping_address = isset( $_POST['shipping_address'] ) ? sanitize_textarea_field( wp_unslash( $_POST['shipping_address'] ) ) : '';
            $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';
            $custom_data = array();
            if ( ! empty( $raffle->form_fields ) ) {
                $decoded_fields = json_decode( $raffle->form_fields, true );
                if ( is_array( $decoded_fields ) ) {
                    foreach ( $decoded_fields as $field ) {
                        $field_name = isset( $field['name'] ) ? $field['name'] : '';
                        if ( $field_name && isset( $_POST[$field_name] ) ) {
                            if ( isset( $field['type'] ) && $field['type'] === 'textarea' ) {
                                $custom_data[$field_name] = sanitize_textarea_field( wp_unslash( $_POST[$field_name] ) );
                            } else {
                                $custom_data[$field_name] = sanitize_text_field( wp_unslash( $_POST[$field_name] ) );
                            }
                        }
                    }
                }
            }
            $data = array(
                'name'             => $name,
                'phone'            => $phone,
                'shipping_address' => $shipping_address,
                'email'            => $email,
                'custom_data'      => ! empty( $custom_data ) ? wp_json_encode( $custom_data ) : '',
            );
            $wpdb->update( $this->entries_table, $data, array( 'id' => $entry_id ) );
            echo '<div class="notice notice-success"><p>Entry updated successfully.</p></div>';
            $entry = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->entries_table} WHERE id = %d", $entry_id ) );
        }
        $current_custom = array();
        if ( ! empty( $entry->custom_data ) ) {
            $current_custom = json_decode( $entry->custom_data, true );
        }
        ?>
        <div class="wrap">
            <h1>Edit Entry</h1>
            <form method="post">
                <?php wp_nonce_field( 'edit_entry_' . $entry_id, 'edit_entry_nonce' ); ?>
                <table class="form-table">
                    <tr>
                        <th><label for="name">Name</label></th>
                        <td><input type="text" name="name" id="name" value="<?php echo esc_attr( $entry->name ); ?>" required></td>
                    </tr>
                    <tr>
                        <th><label for="phone">Phone</label></th>
                        <td><input type="text" name="phone" id="phone" value="<?php echo esc_attr( $entry->phone ); ?>"></td>
                    </tr>
                    <tr>
                        <th><label for="shipping_address">Shipping Address</label></th>
                        <td><textarea name="shipping_address" id="shipping_address" required><?php echo esc_textarea( $entry->shipping_address ); ?></textarea></td>
                    </tr>
                    <tr>
                        <th><label for="email">Email</label></th>
                        <td><input type="email" name="email" id="email" value="<?php echo esc_attr( $entry->email ); ?>" required></td>
                    </tr>
        <?php
        if ( ! empty( $raffle->form_fields ) ) {
            $decoded_fields = json_decode( $raffle->form_fields, true );
            if ( is_array( $decoded_fields ) ) {
                foreach ( $decoded_fields as $field ) {
                    $field_name = isset( $field['name'] ) ? $field['name'] : '';
                    $field_label = isset( $field['label'] ) ? $field['label'] : $field_name;
                    $field_type = isset( $field['type'] ) ? $field['type'] : 'text';
                    $current_value = isset( $current_custom[$field_name] ) ? $current_custom[$field_name] : '';
                    ?>
                    <tr>
                        <th><label for="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_html( $field_label ); ?></label></th>
                        <td>
                            <?php if ( $field_type === 'textarea' ): ?>
                                <textarea name="<?php echo esc_attr( $field_name ); ?>" id="<?php echo esc_attr( $field_name ); ?>"><?php echo esc_textarea( $current_value ); ?></textarea>
                            <?php elseif ( $field_type === 'select' && isset( $field['options'] ) && is_array( $field['options'] ) ): ?>
                                <select name="<?php echo esc_attr( $field_name ); ?>" id="<?php echo esc_attr( $field_name ); ?>">
                                    <?php foreach ( $field['options'] as $option ): ?>
                                        <option value="<?php echo esc_attr( $option ); ?>" <?php selected( $current_value, $option ); ?>><?php echo esc_html( $option ); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            <?php else: ?>
                                <input type="text" name="<?php echo esc_attr( $field_name ); ?>" id="<?php echo esc_attr( $field_name ); ?>" value="<?php echo esc_attr( $current_value ); ?>">
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php
                }
            }
        }
        ?>
                </table>
                <?php submit_button( 'Update Entry' ); ?>
            </form>
            <a href="<?php echo admin_url( 'admin.php?page=simple-raffles-view&id=' . $entry->raffle_id ); ?>">Back to Results</a>
        </div>
        <?php
    }

    // ---------------------
    // Admin: Delete Raffle Page
    // ---------------------
    public function admin_delete_raffle_page() {
        global $wpdb;
        if ( ! isset( $_GET['id'] ) ) {
            echo '<div class="wrap"><h1>No raffle drawing specified.</h1></div>';
            return;
        }
        $raffle_id = intval( $_GET['id'] );
        $raffle = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->drawings_table} WHERE id = %d", $raffle_id ) );
        if ( ! $raffle ) {
            echo '<div class="wrap"><h1>Raffle not found.</h1></div>';
            return;
        }
        $nonce = isset( $_GET['raffle_nonce'] ) ? $_GET['raffle_nonce'] : '';
        if ( ! wp_verify_nonce( $nonce, 'delete_raffle_' . $raffle_id ) ) {
            echo '<div class="wrap"><h1>Invalid nonce.</h1></div>';
            return;
        }
        if ( $_SERVER['REQUEST_METHOD'] === 'POST' ) {
            $wpdb->delete( $this->drawings_table, array( 'id' => $raffle_id ) );
            $wpdb->delete( $this->entries_table, array( 'raffle_id' => $raffle_id ) );
            wp_redirect( admin_url( 'admin.php?page=simple-raffles&raffle_deleted=1' ) );
            exit;
        }
        ?>
        <div class="wrap">
            <h1>Delete Raffle Drawing</h1>
            <p>Are you sure you want to delete the raffle drawing: <strong><?php echo esc_html( $raffle->title ); ?></strong>?</p>
            <form method="post">
                <?php submit_button( 'Yes, Delete Raffle', 'delete' ); ?>
                <a href="<?php echo admin_url( 'admin.php?page=simple-raffles' ); ?>">Cancel</a>
            </form>
        </div>
        <?php
    }

    // ---------------------
    // Admin: Reset Raffle Page
    // ---------------------
    public function admin_reset_raffle_page() {
        global $wpdb;
        if ( ! isset( $_GET['id'] ) ) {
            echo '<div class="wrap"><h1>No raffle drawing specified.</h1></div>';
            return;
        }
        $raffle_id = intval( $_GET['id'] );
        $raffle = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->drawings_table} WHERE id = %d", $raffle_id ) );
        if ( ! $raffle ) {
            echo '<div class="wrap"><h1>Raffle not found.</h1></div>';
            return;
        }
        $nonce = isset( $_GET['raffle_nonce'] ) ? $_GET['raffle_nonce'] : '';
        if ( ! wp_verify_nonce( $nonce, 'reset_raffle_' . $raffle_id ) ) {
            echo '<div class="wrap"><h1>Invalid nonce.</h1></div>';
            return;
        }
        if ( $_SERVER['REQUEST_METHOD'] === 'POST' ) {
            $wpdb->delete( $this->entries_table, array( 'raffle_id' => $raffle_id ) );
            $wpdb->update( $this->drawings_table, array( 'winner_entry_id' => null ), array( 'id' => $raffle_id ) );
            wp_redirect( admin_url( 'admin.php?page=simple-raffles&raffle_reset=1' ) );
            exit;
        }
        ?>
        <div class="wrap">
            <h1>Reset Raffle Drawing</h1>
            <p>Are you sure you want to reset the entries for raffle drawing: <strong><?php echo esc_html( $raffle->title ); ?></strong>? This will delete all entries and reopen the raffle.</p>
            <form method="post">
                <?php submit_button( 'Yes, Reset Raffle' ); ?>
                <a href="<?php echo admin_url( 'admin.php?page=simple-raffles' ); ?>">Cancel</a>
            </form>
        </div>
        <?php
    }

    // ---------------------
    // Admin: Remove Winner Page (Reopen Raffle)
    // ---------------------
    public function admin_remove_winner_page() {
        global $wpdb;
        if ( ! isset( $_GET['id'] ) ) {
            echo '<div class="wrap"><h1>No raffle drawing specified.</h1></div>';
            return;
        }
        $raffle_id = intval( $_GET['id'] );
        $raffle = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->drawings_table} WHERE id = %d", $raffle_id ) );
        if ( ! $raffle ) {
            echo '<div class="wrap"><h1>Raffle not found.</h1></div>';
            return;
        }
        $nonce = isset( $_GET['raffle_nonce'] ) ? $_GET['raffle_nonce'] : '';
        if ( ! wp_verify_nonce( $nonce, 'remove_winner_' . $raffle_id ) ) {
            echo '<div class="wrap"><h1>Invalid nonce.</h1></div>';
            return;
        }
        if ( $_SERVER['REQUEST_METHOD'] === 'POST' ) {
            $wpdb->update( $this->drawings_table, array( 'winner_entry_id' => null ), array( 'id' => $raffle_id ) );
            wp_redirect( admin_url( 'admin.php?page=simple-raffles&winner_removed=1' ) );
            exit;
        }
        ?>
        <div class="wrap">
            <h1>Remove Winner</h1>
            <p>Are you sure you want to remove the winner from this raffle? This will reopen the raffle for new entries while keeping existing entries.</p>
            <form method="post">
                <?php submit_button( 'Yes, Remove Winner' ); ?>
                <a href="<?php echo admin_url( 'admin.php?page=simple-raffles' ); ?>">Cancel</a>
            </form>
        </div>
        <?php
    }

    // ---------------------
    // Admin: View Entries Page (Auto Select Winner + Show Custom Data and Delete/Edit Action)
    // ---------------------
    public function admin_view_entries_page() {
        global $wpdb;
        if ( ! isset( $_GET['id'] ) ) {
            echo '<div class="wrap"><h1>No raffle drawing specified.</h1></div>';
            return;
        }
        $raffle_id = intval( $_GET['id'] );

        if ( isset( $_GET['delete_entry'] ) ) {
            $delete_entry_id = intval( $_GET['delete_entry'] );
            $nonce = isset( $_GET['entry_nonce'] ) ? $_GET['entry_nonce'] : '';
            if ( wp_verify_nonce( $nonce, 'delete_entry_' . $delete_entry_id ) ) {
                $wpdb->delete( $this->entries_table, array( 'id' => $delete_entry_id ) );
                wp_redirect( remove_query_arg( array( 'delete_entry', 'entry_nonce' ), $_SERVER['REQUEST_URI'] ) );
                exit;
            } else {
                echo '<div class="notice notice-error"><p>Invalid nonce for deleting entry.</p></div>';
            }
        }

        if ( $_SERVER['REQUEST_METHOD'] === 'POST' && isset( $_POST['select_winner_nonce'] ) && wp_verify_nonce( $_POST['select_winner_nonce'], 'select_winner_' . $raffle_id ) ) {
            $winner = $wpdb->get_row(
                $wpdb->prepare(
                    "SELECT * FROM {$this->entries_table} WHERE raffle_id = %d ORDER BY RAND() LIMIT 1",
                    $raffle_id
                )
            );
            if ( $winner ) {
                $wpdb->update( $this->drawings_table, array( 'winner_entry_id' => $winner->id ), array( 'id' => $raffle_id ) );
            }
        }

        $drawing = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->drawings_table} WHERE id = %d", $raffle_id ) );
        $entries = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$this->entries_table} WHERE raffle_id = %d", $raffle_id ) );
        ?>
        <div class="wrap">
            <h1>Results for: <?php echo esc_html( $drawing->title ); ?></h1>
            <?php
            if ( ! empty( $drawing->winner_entry_id ) ) {
                $winner = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->entries_table} WHERE id = %d", $drawing->winner_entry_id ) );
                if ( $winner ) {
                    echo '<div style="background:#dff0d8; padding:10px; margin-bottom:20px;">';
                    echo '<strong>Winner Selected!</strong><br>';
                    echo 'Name: ' . esc_html( $winner->name ) . '<br>';
                    echo 'Email: ' . esc_html( $winner->email ) . '<br>';
                    echo 'Shipping Address: ' . esc_html( $winner->shipping_address ) . '<br>';

                    $form_fields = json_decode( $drawing->form_fields, true );
                    $winner_custom = json_decode( $winner->custom_data, true );
                    if ( is_array( $form_fields ) && ! empty( $winner_custom ) ) {
                        echo '<strong>Custom Data:</strong><br>';
                        foreach ( $form_fields as $field ) {
                            $field_name = isset( $field['name'] ) ? $field['name'] : '';
                            $field_label = isset( $field['label'] ) ? $field['label'] : $field_name;
                            $value = ( isset( $winner_custom[$field_name] ) && $winner_custom[$field_name] !== '' ) ? $winner_custom[$field_name] : '';
                            if ( $value !== '' ) {
                                echo esc_html( $field_label . ': ' . $value ) . '<br>';
                            }
                        }
                    }
                    echo '</div>';
                }
            } else {
                ?>
                <form method="post" style="margin-bottom:20px;">
                    <?php wp_nonce_field( 'select_winner_' . $raffle_id, 'select_winner_nonce' ); ?>
                    <input type="submit" value="Auto Select Winner" class="button button-primary">
                </form>
                <?php
            }
            ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Name</th>
                        <th>Phone</th>
                        <th>Shipping Address</th>
                        <th>Email</th>
                        <th>Custom Data</th>
                        <th>Entry Date</th>
                        <th>Actions</th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ( $entries ) : ?>
                        <?php foreach ( $entries as $entry ) : ?>
                            <tr>
                                <td><?php echo esc_html( $entry->id ); ?></td>
                                <td><?php echo esc_html( $entry->name ); ?></td>
                                <td><?php echo esc_html( $entry->phone ); ?></td>
                                <td><?php echo esc_html( $entry->shipping_address ); ?></td>
                                <td><?php echo esc_html( $entry->email ); ?></td>
                                <td>
                                    <?php
                                    if ( ! empty( $entry->custom_data ) ) {
                                        $data = json_decode( $entry->custom_data, true );
                                        if ( is_array( $data ) ) {
                                            foreach ( $data as $key => $value ) {
                                                echo esc_html( $key . ': ' . $value ) . '<br>';
                                            }
                                        } else {
                                            echo esc_html( $entry->custom_data );
                                        }
                                    }
                                    ?>
                                </td>
                                <td><?php echo esc_html( $entry->entry_date ); ?></td>
                                <td>
                                    <a href="<?php echo admin_url( 'admin.php?page=simple-raffles-edit-entry&entry_id=' . $entry->id . '&entry_nonce=' . wp_create_nonce( 'edit_entry_' . $entry->id ) ); ?>">Edit</a> | 
                                    <a href="<?php echo admin_url( 'admin.php?page=simple-raffles-view&id=' . $raffle_id . '&delete_entry=' . $entry->id . '&entry_nonce=' . wp_create_nonce( 'delete_entry_' . $entry->id ) ); ?>">Delete</a>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php else : ?>
                        <tr><td colspan="8">No entries found.</td></tr>
                    <?php endif; ?>
                </tbody>
            </table>
            <br>
            <a href="<?php echo admin_url( 'admin.php?page=simple-raffles' ); ?>">Back to Raffle Drawings</a>
        </div>
        <?php
    }

    // ---------------------
    // Front-End Raffle Entry Form Shortcode
    // ---------------------
    public function raffle_shortcode( $atts ) {
        $atts = shortcode_atts( array(
            'id' => 0,
        ), $atts, 'raffle' );

        $raffle_id = intval( $atts['id'] );
        if ( ! $raffle_id ) {
            return 'Raffle drawing not found.';
        }
        global $wpdb;
        $raffle = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->drawings_table} WHERE id = %d", $raffle_id ) );

        if ( ! empty( $raffle->winner_entry_id ) ) {
            $winner = $wpdb->get_row( $wpdb->prepare( "SELECT name FROM {$this->entries_table} WHERE id = %d", $raffle->winner_entry_id ) );
            if ( $winner ) {
                $options = get_option( 'simple_raffle_options' );
                $winner_display_text = isset( $options['winner_display_text'] ) ? $options['winner_display_text'] : 'Winner:';
                $winner_display_font_size = isset( $options['winner_display_font_size'] ) ? $options['winner_display_font_size'] : '20px';
                $winner_display_color = isset( $options['winner_display_color'] ) ? $options['winner_display_color'] : '#000000';
                ob_start();
                ?>
                <div class="simple-raffle-winner" style="font-size: <?php echo esc_attr( $winner_display_font_size ); ?>; color: <?php echo esc_attr( $winner_display_color ); ?>; padding: 10px;">
                    <?php echo esc_html( $winner_display_text . ' ' . $winner->name ); ?>
                </div>
                <?php
                return ob_get_clean();
            }
        }

        $message = '';
        if ( isset( $_GET['raffle_status'] ) ) {
            if ( $_GET['raffle_status'] === 'success' ) {
                $message = '<p style="color:green;">Your entry was submitted successfully.</p>';
            } elseif ( $_GET['raffle_status'] === 'duplicate' ) {
                $message = '<p style="color:red;">You have already entered this raffle.</p>';
            }
        }

        $options = get_option( 'simple_raffle_options' );
        $form_title_font_size = isset( $options['form_title_font_size'] ) ? $options['form_title_font_size'] : '24px';
        $form_title_color = isset( $options['form_title_color'] ) ? $options['form_title_color'] : '#333333';
        $field_font_size = isset( $options['field_font_size'] ) ? $options['field_font_size'] : '16px';
        $field_color = isset( $options['field_color'] ) ? $options['field_color'] : '#333333';
        $form_background_color = isset( $options['form_background_color'] ) ? $options['form_background_color'] : '#f9f9f9';
        $submit_button_text = isset( $options['submit_button_text'] ) ? $options['submit_button_text'] : 'Enter Raffle';
        $submit_button_text_color = isset( $options['submit_button_text_color'] ) ? $options['submit_button_text_color'] : '#ffffff';
        $submit_button_background_color = isset( $options['submit_button_background_color'] ) ? $options['submit_button_background_color'] : '#0073aa';
        $submit_button_font_size = isset( $options['submit_button_font_size'] ) ? $options['submit_button_font_size'] : '16px';

        $custom_fields = array();
        if ( ! empty( $raffle->form_fields ) ) {
            $decoded = json_decode( $raffle->form_fields, true );
            if ( is_array( $decoded ) ) {
                $custom_fields = $decoded;
            }
        }

        ob_start();
        ?>
        <div class="simple-raffle-form" style="background-color: <?php echo esc_attr( $form_background_color ); ?>; padding: 10px;">
            <?php echo $message; ?>
            <h2 style="font-size: <?php echo esc_attr( $form_title_font_size ); ?>; color: <?php echo esc_attr( $form_title_color ); ?>;">Enter Raffle: <?php echo esc_html( $raffle->title ); ?></h2>
            <form method="post">
                <?php wp_nonce_field( 'submit_raffle_' . $raffle_id, 'raffle_nonce' ); ?>
                <input type="hidden" name="raffle_id" value="<?php echo esc_attr( $raffle_id ); ?>">
                <p>
                    <label for="raffle_name" style="font-size: <?php echo esc_attr( $field_font_size ); ?>; color: <?php echo esc_attr( $field_color ); ?>;">Name <span style="color:red">*</span></label><br>
                    <input type="text" id="raffle_name" name="name" required style="font-size: <?php echo esc_attr( $field_font_size ); ?>;">
                </p>
                <p>
                    <label for="raffle_phone" style="font-size: <?php echo esc_attr( $field_font_size ); ?>; color: <?php echo esc_attr( $field_color ); ?>;">Phone</label><br>
                    <input type="text" id="raffle_phone" name="phone" style="font-size: <?php echo esc_attr( $field_font_size ); ?>;">
                </p>
                <p>
                    <label for="raffle_shipping" style="font-size: <?php echo esc_attr( $field_font_size ); ?>; color: <?php echo esc_attr( $field_color ); ?>;">Shipping Address <span style="color:red">*</span></label><br>
                    <textarea id="raffle_shipping" name="shipping_address" rows="3" required style="font-size: <?php echo esc_attr( $field_font_size ); ?>;"></textarea>
                </p>
                <p>
                    <label for="raffle_email" style="font-size: <?php echo esc_attr( $field_font_size ); ?>; color: <?php echo esc_attr( $field_color ); ?>;">Email <span style="color:red">*</span></label><br>
                    <input type="email" id="raffle_email" name="email" required style="font-size: <?php echo esc_attr( $field_font_size ); ?>;">
                </p>
                <?php if ( ! empty( $custom_fields ) ) : ?>
                    <?php foreach ( $custom_fields as $field ) : 
                        $field_name = ! empty( $field['name'] ) ? $field['name'] : 'custom_field_' . uniqid();
                        $field_label = ! empty( $field['label'] ) ? $field['label'] : 'Custom Field';
                        $field_type = ! empty( $field['type'] ) ? $field['type'] : 'text';
                        $required = ( ! empty( $field['required'] ) && $field['required'] ) ? 'required' : '';
                        $placeholder = ! empty( $field['placeholder'] ) ? $field['placeholder'] : '';
                        ?>
                        <p>
                            <label for="<?php echo esc_attr( $field_name ); ?>" style="font-size: <?php echo esc_attr( $field_font_size ); ?>; color: <?php echo esc_attr( $field_color ); ?>;">
                                <?php echo esc_html( $field_label ); ?> <?php echo $required ? '<span style="color:red">*</span>' : ''; ?>
                            </label><br>
                            <?php if ( $field_type === 'textarea' ) : ?>
                                <textarea id="<?php echo esc_attr( $field_name ); ?>" name="<?php echo esc_attr( $field_name ); ?>" rows="3" placeholder="<?php echo esc_attr( $placeholder ); ?>" style="font-size: <?php echo esc_attr( $field_font_size ); ?>;" <?php echo $required; ?>></textarea>
                            <?php elseif ( $field_type === 'select' && isset( $field['options'] ) && is_array( $field['options'] ) ) : ?>
                                <select id="<?php echo esc_attr( $field_name ); ?>" name="<?php echo esc_attr( $field_name ); ?>" style="font-size: <?php echo esc_attr( $field_font_size ); ?>;" <?php echo $required; ?>>
                                    <option value="">-- Select --</option>
                                    <?php foreach ( $field['options'] as $option ) : ?>
                                        <option value="<?php echo esc_attr( $option ); ?>"><?php echo esc_html( $option ); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            <?php else : ?>
                                <input type="text" id="<?php echo esc_attr( $field_name ); ?>" name="<?php echo esc_attr( $field_name ); ?>" placeholder="<?php echo esc_attr( $placeholder ); ?>" style="font-size: <?php echo esc_attr( $field_font_size ); ?>;" <?php echo $required; ?>>
                            <?php endif; ?>
                        </p>
                    <?php endforeach; ?>
                <?php endif; ?>
                <p>
                    <input type="submit" value="<?php echo esc_attr( $submit_button_text ); ?>" 
                           style="font-size: <?php echo esc_attr( $submit_button_font_size ); ?>; 
                                  color: <?php echo esc_attr( $submit_button_text_color ); ?>; 
                                  background-color: <?php echo esc_attr( $submit_button_background_color ); ?>;">
                </p>
            </form>
        </div>
        <?php
        return ob_get_clean();
    }

    // ---------------------
    // Handle Front-End Submission (Store custom field data in custom_data)
    // ---------------------
    public function handle_entry_submission() {
        if ( $_SERVER['REQUEST_METHOD'] === 'POST' && isset( $_POST['raffle_id'] ) && isset( $_POST['raffle_nonce'] ) ) {
            $raffle_id = intval( $_POST['raffle_id'] );
            if ( ! wp_verify_nonce( $_POST['raffle_nonce'], 'submit_raffle_' . $raffle_id ) ) {
                return;
            }
            global $wpdb;

            $raffle = $wpdb->get_row( $wpdb->prepare( "SELECT winner_entry_id, form_fields FROM {$this->drawings_table} WHERE id = %d", $raffle_id ) );
            if ( ! empty( $raffle->winner_entry_id ) ) {
                return;
            }

            $name = isset( $_POST['name'] ) ? sanitize_text_field( wp_unslash( $_POST['name'] ) ) : '';
            $phone = isset( $_POST['phone'] ) ? sanitize_text_field( wp_unslash( $_POST['phone'] ) ) : '';
            $shipping_address = isset( $_POST['shipping_address'] ) ? sanitize_textarea_field( wp_unslash( $_POST['shipping_address'] ) ) : '';
            $email = isset( $_POST['email'] ) ? sanitize_email( wp_unslash( $_POST['email'] ) ) : '';

            if ( empty( $name ) || empty( $shipping_address ) || empty( $email ) ) {
                return;
            }

            $existing = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM {$this->entries_table} WHERE raffle_id = %d AND email = %s", $raffle_id, $email ) );
            if ( $existing ) {
                wp_redirect( add_query_arg( array( 'raffle_status' => 'duplicate' ), get_permalink() ) );
                exit;
            }

            $custom_data = array();
            if ( ! empty( $raffle->form_fields ) ) {
                $decoded_fields = json_decode( $raffle->form_fields, true );
                if ( is_array( $decoded_fields ) ) {
                    foreach ( $decoded_fields as $field ) {
                        $field_name = isset( $field['name'] ) ? $field['name'] : '';
                        if ( $field_name && isset( $_POST[$field_name] ) ) {
                            if ( isset( $field['type'] ) && $field['type'] === 'textarea' ) {
                                $custom_data[$field_name] = sanitize_textarea_field( wp_unslash( $_POST[$field_name] ) );
                            } else {
                                $custom_data[$field_name] = sanitize_text_field( wp_unslash( $_POST[$field_name] ) );
                            }
                        }
                    }
                }
            }

            $wpdb->insert(
                $this->entries_table,
                array(
                    'raffle_id'        => $raffle_id,
                    'name'             => $name,
                    'phone'            => $phone,
                    'shipping_address' => $shipping_address,
                    'email'            => $email,
                    'custom_data'      => ! empty( $custom_data ) ? wp_json_encode( $custom_data ) : '',
                )
            );

            wp_redirect( add_query_arg( array( 'raffle_status' => 'success' ), get_permalink() ) );
            exit;
        }
    }
}

// Initialize the plugin.
new Simple_Raffle_Plugin();
