<?php
if (!defined('ABSPATH')) { exit; }

require_once __DIR__ . '/../rest/request.php';

function flw_su_assert_manage_options_and_nonce() {
	if (!current_user_can('manage_options')) {
		wp_send_json_error(['message' => 'Unauthorized'], 403);
	}
	check_ajax_referer('flw_su_admin', 'nonce');
}

add_action('wp_ajax_flw_su_exchange_code', function(){
	flw_su_assert_manage_options_and_nonce();
	$code = isset($_POST['code']) ? sanitize_text_field($_POST['code']) : '';
	if ($code === '') { wp_send_json_error(['message' => 'Missing code'], 400); }
	$site_url = home_url();
	$contact_email = get_bloginfo('admin_email');
	$broker = defined('FLW_SU_BROKER_BASE') ? FLW_SU_BROKER_BASE : esc_url_raw(get_option(FLW_SU_OPTION_BROKER_URL, ''));
	$endpoint = rtrim($broker, '/') . '/wp-json/flw_su/v1/register/exchange';
	$args = [
		'headers' => [ 'Content-Type' => 'application/json' ],
		'body' => wp_json_encode([ 'code' => $code, 'site_url' => $site_url, 'contact_email' => $contact_email ]),
		'timeout' => 15,
	];
	$response = wp_remote_post($endpoint, $args);
	if (is_wp_error($response)) {
		wp_send_json_error(['message' => $response->get_error_message()], 500);
	}
	$codeResp = (int) wp_remote_retrieve_response_code($response);
	$body = (string) wp_remote_retrieve_body($response);
	$data = json_decode($body, true);
	if ($codeResp !== 200 || !is_array($data) || empty($data['client_id']) || empty($data['client_secret'])) {
		wp_send_json_error(['message' => 'Code exchange failed', 'status' => $codeResp, 'response' => $body], 500);
	}
	update_option(FLW_SU_OPTION_CLIENT_ID, sanitize_text_field($data['client_id']));
	update_option(FLW_SU_OPTION_CLIENT_SECRET, sanitize_text_field($data['client_secret']));
	wp_send_json_success(['client_id' => $data['client_id']]);
});

add_action('wp_ajax_flw_su_register_site', function () {
	flw_su_assert_manage_options_and_nonce();
	$broker = defined('FLW_SU_BROKER_BASE') ? FLW_SU_BROKER_BASE : esc_url_raw(get_option(FLW_SU_OPTION_BROKER_URL, ''));
	$site_url = home_url();
	$contact_email = get_bloginfo('admin_email');
	$endpoint = rtrim($broker, '/') . '/wp-json/flw_su/v1/register-site';
	$args = [
		'headers' => [ 'Content-Type' => 'application/json' ],
		'body' => wp_json_encode([ 'site_url' => $site_url, 'contact_email' => $contact_email ]),
		'timeout' => 15,
	];
	$response = wp_remote_post($endpoint, $args);
	if (is_wp_error($response)) {
		wp_send_json_error(['message' => $response->get_error_message()], 500);
	}
	$code = wp_remote_retrieve_response_code($response);
	$body = wp_remote_retrieve_body($response);
	$data = json_decode($body, true);
	if ($code !== 200 || !is_array($data) || empty($data['client_id']) || empty($data['client_secret'])) {
		wp_send_json_error(['message' => 'Broker registration failed', 'status' => $code, 'response' => $body], 500);
	}
	update_option(FLW_SU_OPTION_CLIENT_ID, sanitize_text_field($data['client_id']));
	update_option(FLW_SU_OPTION_CLIENT_SECRET, sanitize_text_field($data['client_secret']));
	wp_send_json_success(['client_id' => $data['client_id']]);
});

add_action('wp_ajax_flw_su_clear_registration', function () {
	flw_su_assert_manage_options_and_nonce();
	$client_id = get_option(FLW_SU_OPTION_CLIENT_ID, '');
	if (!empty($client_id)) {
		$resp = flw_su_remote_request('POST', '/flw_su/v1/clients/revoke', []);
	}
	update_option(FLW_SU_OPTION_CLIENT_ID, '');
	update_option(FLW_SU_OPTION_CLIENT_SECRET, '');
	wp_send_json_success(['cleared' => true]);
});

add_action('wp_ajax_flw_su_status', function () {
	flw_su_assert_manage_options_and_nonce();
	$client_id = get_option(FLW_SU_OPTION_CLIENT_ID, '');
	if (empty($client_id)) {
		wp_send_json_success(['status' => 'unregistered', 'client_id' => '']);
	}
	$result = flw_su_remote_request('GET', '/flw_su/v1/subscriptions/status');
	if (is_wp_error($result)) {
		wp_send_json_error(['message' => $result->get_error_message()], 500);
	}
	if ((int)$result['code'] !== 200) {
		wp_send_json_error(['message' => 'Remote status failed', 'status' => (int)$result['code'], 'response' => $result['body']], 500);
	}
	$status = is_array($result['json']) && isset($result['json']['authorized']) ? ($result['json']['authorized'] ? 'authorized' : 'unauthorized') : 'unknown';
	$has_customer = is_array($result['json']) && !empty($result['json']['has_customer']);
	$plan_key = is_array($result['json']) && is_string($result['json']['plan_key'] ?? null) ? $result['json']['plan_key'] : '';
	$current_period_end = is_array($result['json']) ? ($result['json']['current_period_end'] ?? null) : null;
	$exempt = is_array($result['json']) && !empty($result['json']['exempt']);
	wp_send_json_success([
		'status' => $status,
		'client_id' => $client_id,
		'has_customer' => $has_customer,
		'plan_key' => $plan_key,
		'current_period_end' => $current_period_end,
		'exempt' => $exempt,
	]);
});

add_action('wp_ajax_flw_su_checkout_session', function () {
	flw_su_assert_manage_options_and_nonce();
	$plan_key = isset($_POST['plan_key']) ? sanitize_key($_POST['plan_key']) : '';
	$return_url = isset($_POST['return_url']) ? esc_url_raw((string)$_POST['return_url']) : '';
	$body = $plan_key ? ['plan_key' => $plan_key] : [];
	if ($return_url) { $body['return_url'] = $return_url; }
	$result = flw_su_remote_request('POST', '/flw_su/v1/subscriptions/checkout/session', $body);
	if (is_wp_error($result)) {
		wp_send_json_error(['message' => $result->get_error_message()], 500);
	}
	if ((int)$result['code'] !== 200) {
		$message = 'Remote checkout failed';
		if (is_array($result['json']) && !empty($result['json']['message'])) {
			$message = (string)$result['json']['message'];
		} elseif (!empty($result['body'])) {
			$message = 'Remote checkout failed: ' . substr((string)$result['body'], 0, 200);
		}
		wp_send_json_error(['message' => $message, 'status' => (int)$result['code']], 500);
	}
	$url = is_array($result['json']) && !empty($result['json']['url']) ? $result['json']['url'] : '';
	wp_send_json_success(['url' => $url]);
});

add_action('wp_ajax_flw_su_billing_portal', function () {
	flw_su_assert_manage_options_and_nonce();
	$result = flw_su_remote_request('POST', '/flw_su/v1/subscriptions/portal', []);
	if (is_wp_error($result)) {
		wp_send_json_error(['message' => $result->get_error_message()], 500);
	}
	if ((int)$result['code'] !== 200) {
		wp_send_json_error(['message' => 'Remote portal failed', 'status' => (int)$result['code']], 500);
	}
	$url = is_array($result['json']) && !empty($result['json']['url']) ? $result['json']['url'] : '';
	wp_send_json_success(['url' => $url]);
});

add_action('wp_ajax_flw_su_product_example', function () {
	flw_su_assert_manage_options_and_nonce();
	$result = flw_su_remote_request('POST', '/flw_su/v1/product/example/action', []);
	if (is_wp_error($result)) {
		wp_send_json_error(['message' => $result->get_error_message()], 500);
	}
	$code = (int) $result['code'];
	if ($code !== 200) {
		$message = 'Remote example action failed';
		if (is_array($result['json']) && !empty($result['json']['message'])) {
			$message = $result['json']['message'];
		}
		wp_send_json_error(['message' => $message, 'status' => $code], $code);
	}
	wp_send_json_success($result['json']);
});

// Run WordPress update for a specific plugin file if subscriber is authorized
add_action('wp_ajax_flw_su_run_update', function(){
	flw_su_assert_manage_options_and_nonce();
	$plugin = isset($_POST['plugin']) ? sanitize_text_field($_POST['plugin']) : '';
	if ($plugin === '') { wp_send_json_error(['message' => 'Missing plugin'], 400); }
	// Check subscription status
	$status = flw_su_remote_request('GET', '/flw_su/v1/subscriptions/status');
	if (is_wp_error($status)) { wp_send_json_error(['message' => $status->get_error_message()], 500); }
	$authorized = (int)($status['code'] ?? 500) === 200 && !empty($status['json']['authorized']);
	if (!$authorized) {
		wp_send_json_error(['message' => 'Subscription required'], 402);
	}
	// Trigger background update using WP_Upgrader
	if (!class_exists('Plugin_Upgrader')) { require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php'; }
	if (!function_exists('wp_update_plugins')) { require_once ABSPATH . 'wp-includes/update.php'; }
	wp_update_plugins();
	$skin = new Automatic_Upgrader_Skin();
	$upgrader = new Plugin_Upgrader($skin);
	$result = $upgrader->upgrade($plugin);
	if (is_wp_error($result)) {
		wp_send_json_error(['message' => $result->get_error_message()], 500);
	}
	if ($result === false) {
		wp_send_json_error(['message' => 'Update failed or no update available'], 500);
	}
	wp_send_json_success(['updated' => true]);
});