<?php
if (!defined('ABSPATH')) { exit; }

/* =======================================
   FLW Plugin Library Admin Menu & Settings
   ======================================= */
if ( ! class_exists( 'FLW_Plugin_Library' ) ) {
	class FLW_Plugin_Library {
		public static function initialize() {
			add_action( 'admin_init', [ self::class, 'register_settings' ] );
			add_action( 'admin_menu', [ self::class, 'add_main_menu' ] );
            // When Facebook app credentials change, refresh the app access token
            add_action( 'updated_option', [ self::class, 'maybe_regenerate_facebook_token' ], 10, 1 );
            add_action( 'added_option',   [ self::class, 'maybe_regenerate_facebook_token' ], 10, 1 );
            // Handle Debug log reset action
            add_action( 'admin_post_flw_reset_debug_log', [ self::class, 'handle_reset_debug_log' ] );
            // AJAX: Reset debug log without redirect
            add_action( 'wp_ajax_flw_reset_debug_log', [ self::class, 'ajax_reset_debug_log' ] );
		}

		public static function add_main_menu() {
			if ( ! has_action( 'flw_plugins_menu_created' ) ) {
				add_menu_page(
					'FLW Plugins',
					'FLW Plugins',
					'manage_options',
					'flw-plugins',
					[ self::class, 'render_main_page' ],
					'dashicons-admin-generic',
					-80
				);
				do_action( 'flw_plugins_menu_created' );
			}
			add_submenu_page(
				'flw-plugins',
				'Global Settings',
				'Global Settings',
				'manage_options',
				'flw-global-settings',
				[ self::class, 'render_settings_page' ]
			);
		}

		public static function add_submenu( $title, $slug, $callback ) {
			add_submenu_page(
				'flw-plugins',
				$title,
				$title,
				'manage_options',
				$slug,
				$callback
			);
		}

		public static function render_main_page() {
			$subscribeUrl = admin_url('admin.php?page=flw_su_settings&tab=subscriptions');
			$updatesUrl   = 'https://frostlineworks.com/updates/';
			?>
			<div class="wrap">
				<h1 style="margin-bottom:16px;">Secure Updates by FLW — Subscribe for One‑Click Upgrades</h1>
				<div style="margin:14px 0 22px; padding:24px; border:3px solid #2271b1; border-radius:10px; background:linear-gradient(135deg,#f0f6fc 0%, #ffffff 60%, #ffffff 100%);">
					<div style="display:flex; align-items:flex-start; gap:24px; flex-wrap:wrap;">
						<div style="flex:1; min-width:260px;">
							<h2 style="margin:0 0 8px; font-size:22px; line-height:1.3; color:#1d2327;">Stop manual downloads. Keep your FLW plugins securely up‑to‑date.</h2>
							<p style="margin:0 0 12px; font-size:14px; color:#2c3338; max-width:720px;">
								With an active FLW subscription, updates flow directly into WordPress. No zip files, no upload steps—just click Update.
							</p>
							<div style="display:flex; gap:10px; flex-wrap:wrap;">
								<a class="button button-primary" style="height:auto; padding:10px 16px; font-weight:600;" href="<?php echo esc_url( $subscribeUrl ); ?>">View Plans & Subscribe</a>
								<a class="button" style="height:auto; padding:10px 16px;" href="<?php echo esc_url( $updatesUrl ); ?>" target="_blank" rel="noopener noreferrer">Learn More</a>
							</div>
						</div>
						<div style="flex:0 0 280px; min-width:240px; background:#fff; border:1px solid #d0d7de; border-radius:8px; padding:14px;">
							<h3 style="margin:0 0 8px;">Why teams subscribe</h3>
							<ul style="margin:0; padding-left:18px;">
								<li>One‑click updates from Plugins screen</li>
								<li>Priority access to new features & fixes</li>
								<li>Security updates delivered fast</li>
								<li>No SFTP or file uploads needed</li>
							</ul>
						</div>
					</div>
				</div>

				<div style="display:grid; grid-template-columns:repeat(auto-fit,minmax(220px,1fr)); gap:16px; margin:18px 0;">
					<div style="border:1px solid #d0d7de; border-radius:8px; padding:14px; background:#fff;">
						<h3 style="margin-top:0;">Auto‑Updates</h3>
						<p style="margin-bottom:0;">Run updates directly in WordPress. No zip downloads, no manual steps.</p>
					</div>
					<div style="border:1px solid #d0d7de; border-radius:8px; padding:14px; background:#fff;">
						<h3 style="margin-top:0;">Subscription Entitlement</h3>
						<p style="margin-bottom:0;">Only entitled sites get package URLs. Keeps your licenses respected and secure.</p>
					</div>
					<div style="border:1px solid #d0d7de; border-radius:8px; padding:14px; background:#fff;">
						<h3 style="margin-top:0;">Faster Security Patches</h3>
						<p style="margin-bottom:0;">Ship fixes quickly to production without juggling files or FTP access.</p>
					</div>
					<div style="border:1px solid #d0d7de; border-radius:8px; padding:14px; background:#fff;">
						<h3 style="margin-top:0;">Team‑Friendly</h3>
						<p style="margin-bottom:0;">Standardize update workflows across environments and teammates.</p>
					</div>
				</div>

				<div style="margin:22px 0; padding:18px; border:1px solid #d0d7de; border-radius:8px; background:#fff;">
					<h2 style="margin-top:0;">How it works</h2>
					<ol style="margin:0; padding-left:18px;">
						<li>Subscribe to FLW Secure Updates.</li>
						<li>Connect your site in the FLW dashboard.</li>
						<li>Receive update notifications and apply with one click.</li>
					</ol>
				</div>

				<div style="display:grid; grid-template-columns:repeat(auto-fit,minmax(260px,1fr)); gap:16px; margin:18px 0;">
					<div style="border:1px solid #d0d7de; border-radius:8px; padding:16px; background:#f8fafc;">
						<blockquote style="margin:0; font-style:italic; color:#2c3338;">“The update flow is so much faster now. No more zip juggling.”</blockquote>
						<div style="margin-top:8px; color:#50575e;">— Ava Martinez, Agency Owner</div>
					</div>
					<div style="border:1px solid #d0d7de; border-radius:8px; padding:16px; background:#f8fafc;">
						<blockquote style="margin:0; font-style:italic; color:#2c3338;">“We keep our plugins current without giving devs SFTP access.”</blockquote>
						<div style="margin-top:8px; color:#50575e;">— Liam Chen, CTO, SaaS</div>
					</div>
				</div>

				<div style="margin:22px 0; padding:18px; border:1px solid #d0d7de; border-radius:8px; background:#fff;">
					<h2 style="margin-top:0;">FAQs</h2>
					<p><strong>Can I still update manually?</strong><br/>Yes. Subscriptions enable one‑click updates, but you can always install updates manually.</p>
					<p><strong>What happens if my subscription lapses?</strong><br/>Plugins continue working; you’ll lose access to automatic package delivery and new versions.</p>
					<p><strong>Do you support multisite?</strong><br/>Yes—updates work with standard WordPress multisite configurations.</p>
				</div>

				<div style="margin:26px 0 10px;">
					<a class="button button-primary" style="height:auto; padding:12px 18px; font-weight:600; font-size:14px;" href="<?php echo esc_url( $subscribeUrl ); ?>">View Plans & Subscribe</a>
					<a class="button" style="height:auto; padding:12px 18px; margin-left:8px;" href="<?php echo esc_url( $updatesUrl ); ?>" target="_blank" rel="noopener noreferrer">Learn More</a>
				</div>
			</div>
			<?php
		}

		public static function register_settings() {
			register_setting( 'flw_settings_group', 'flw_stripe_public_key' );
			register_setting( 'flw_settings_group', 'flw_stripe_secret_key' );
			// Stripe mode-aware settings
			register_setting( 'flw_settings_group', 'flw_stripe_mode_live' );
			register_setting( 'flw_settings_group', 'flw_stripe_test_public_key' );
			register_setting( 'flw_settings_group', 'flw_stripe_test_secret_key' );
			register_setting( 'flw_settings_group', 'flw_stripe_live_public_key' );
			register_setting( 'flw_settings_group', 'flw_stripe_live_secret_key' );
			// Facebook app credentials (global)
			register_setting( 'flw_settings_group', 'flw_facebook_app_id' );
			register_setting( 'flw_settings_group', 'flw_facebook_app_secret' );
			register_setting( 'flw_settings_group', 'flw_mail_host' );
			register_setting( 'flw_settings_group', 'flw_mail_port' );
			register_setting( 'flw_settings_group', 'flw_debug_mode' );
			register_setting( 'flw_settings_group', 'flw_transient_duration' );
            // Per‑plugin debug toggles
            register_setting( 'flw_settings_group', 'flw_plugin_debug_toggles', [ 'sanitize_callback' => [ self::class, 'sanitize_plugin_debug_toggles' ] ] );
		}

		/**
		 * Regenerate Facebook App Access Token when credentials are updated.
		 */
        public static function maybe_regenerate_facebook_token( $option, ...$rest ) {
            if ( $option !== 'flw_facebook_app_id' && $option !== 'flw_facebook_app_secret' ) {
                return;
            }
            self::generate_facebook_app_token();
        }

		/**
		 * Generate and store a Facebook App Access Token using app_id and app_secret.
		 */
		private static function generate_facebook_app_token(): void {
			$app_id = (string) get_option( 'flw_facebook_app_id', '' );
			$app_secret = (string) get_option( 'flw_facebook_app_secret', '' );
			if ( $app_id === '' || $app_secret === '' ) {
				return;
			}
			$url = add_query_arg(
				[
					'client_id' => $app_id,
					'client_secret' => $app_secret,
					'grant_type' => 'client_credentials',
				],
				'https://graph.facebook.com/oauth/access_token'
			);
			$resp = wp_remote_get( $url, [ 'timeout' => 10 ] );
			if ( is_wp_error( $resp ) ) {
				set_transient( 'flw_fb_token_error', $resp->get_error_message(), 5 * MINUTE_IN_SECONDS );
				return;
			}
			$code = (int) wp_remote_retrieve_response_code( $resp );
			$body = (string) wp_remote_retrieve_body( $resp );
            if ( $code >= 200 && $code < 300 ) {
                $data = json_decode( $body, true );
                $token = is_array( $data ) && isset( $data['access_token'] ) ? (string) $data['access_token'] : '';
                if ( $token !== '' ) {
                    update_option( 'flw_facebook_app_access_token', $token );
                    update_option( 'flw_facebook_app_token_type', isset( $data['token_type'] ) ? (string) $data['token_type'] : 'bearer' );
                    update_option( 'flw_facebook_app_token_updated', time() );
                    if ( isset( $data['expires_in'] ) ) {
                        update_option( 'flw_facebook_app_token_expires', time() + (int) $data['expires_in'] );
                    } else {
                        delete_option( 'flw_facebook_app_token_expires' );
                    }
                    delete_transient( 'flw_fb_token_error' );
                } else {
                    $message = '';
                    if ( is_array( $data ) && isset( $data['error']['message'] ) ) {
                        $message = (string) $data['error']['message'];
                    } elseif ( is_string( $body ) && $body !== '' ) {
                        $message = 'Unexpected response: ' . substr( $body, 0, 300 );
                    } else {
                        $message = 'Unexpected response without access_token.';
                    }
                    set_transient( 'flw_fb_token_error', $message, 5 * MINUTE_IN_SECONDS );
                }
            } else {
				set_transient( 'flw_fb_token_error', 'HTTP ' . $code . ': ' . $body, 5 * MINUTE_IN_SECONDS );
			}
		}

		/**
		 * Ensure a Facebook App token exists (or is refreshed if expired) when viewing settings.
		 */
		private static function ensure_facebook_app_token_if_needed(): void {
			$app_id = (string) get_option( 'flw_facebook_app_id', '' );
			$app_secret = (string) get_option( 'flw_facebook_app_secret', '' );
			if ( $app_id === '' || $app_secret === '' ) { return; }
			$token   = (string) get_option( 'flw_facebook_app_access_token', '' );
			$expires = (int) get_option( 'flw_facebook_app_token_expires', 0 );
			$needs   = ($token === '') || ($expires > 0 && $expires <= ( time() + 60 ));
			if ( $needs ) { self::generate_facebook_app_token(); }
		}

		public static function render_settings_page() {
			?>
			<div class="wrap">
				<h1>Global Settings</h1>
				<h2 class="nav-tab-wrapper">
					<a href="#stripe" class="nav-tab" onclick="showTab(event,'stripe')">Stripe</a>
					<a href="#facebook" class="nav-tab" onclick="showTab(event,'facebook')">Facebook</a>
					<a href="#debug" class="nav-tab" onclick="showTab(event,'debug')">Debug</a>
					<a href="#instructions" class="nav-tab" onclick="showTab(event,'instructions')">Instructions</a>
				</h2>
				<form method="post" action="options.php">
					<?php settings_fields('flw_settings_group'); ?>
					<div id="instructions" class="tab-content"></div>
					<div id="stripe" class="tab-content">
						<h2>Stripe Settings</h2>
						<div class="flw-toggle-row">
							<label class="flw-switch">
								<input type="checkbox" name="flw_stripe_mode_live" id="flw_stripe_mode_live" value="1" <?php checked( get_option('flw_stripe_mode_live'), 1 ); ?> />
								<span class="flw-slider"></span>
							</label>
							<span class="flw-switch-label"><strong>Live Mode</strong></span>
						</div>
						<p class="description">When Live Mode is enabled, the Live keys will be used; when disabled, the Test keys will be used.</p>

						<div id="flw_stripe_keys_test" class="flw-stripe-keys-group">
							<h3>Test Keys</h3>
							<label>Publishable (Test):
								<input type="text" name="flw_stripe_test_public_key" value="<?php echo esc_attr( get_option('flw_stripe_test_public_key', '') ); ?>" size="50" />
							</label>
							<br/>
							<label>Secret (Test):
								<input type="password" name="flw_stripe_test_secret_key" value="<?php echo esc_attr( get_option('flw_stripe_test_secret_key', '') ); ?>" size="50" />
							</label>
						</div>

						<div id="flw_stripe_keys_live" class="flw-stripe-keys-group">
							<h3>Live Keys</h3>
							<label>Publishable (Live):
								<input type="text" name="flw_stripe_live_public_key" value="<?php echo esc_attr( get_option('flw_stripe_live_public_key', '') ); ?>" size="50" />
							</label>
							<br/>
							<label>Secret (Live):
								<input type="password" name="flw_stripe_live_secret_key" value="<?php echo esc_attr( get_option('flw_stripe_live_secret_key', '') ); ?>" size="50" />
							</label>
						</div>
						<div class="flw-usage">
							<h3 style="margin:0 0 6px;">Using these Stripe settings in other plugins</h3>
							<p style="margin:0 0 8px;">Determine mode and read keys:</p>
							<pre><code id="flw_stripe_usage_code">$useLive   = (bool) get_option('flw_stripe_mode_live');
$publicKey = $useLive ? get_option('flw_stripe_live_public_key', '') : get_option('flw_stripe_test_public_key', '');
$secretKey = $useLive ? get_option('flw_stripe_live_secret_key', '') : get_option('flw_stripe_test_secret_key', '');

// Example (Stripe SDK):
// \\Stripe\\Stripe::setApiKey($secretKey);
// $stripe = new \\Stripe\\StripeClient($secretKey);
</code></pre>
							<button type="button" class="button" onclick="flw_copy_code_by_id('flw_stripe_usage_code', this)">Copy</button>
						</div>
					</div>

					<div id="facebook" class="tab-content">
						<h2>Facebook Settings</h2>
						<p class="description">Store your Facebook App credentials globally so any FLW plugin can reuse them.</p>
						<?php 
							// Attempt generation if missing/expired so the status reflects current state
							self::ensure_facebook_app_token_if_needed();
							$__fb_token   = (string) get_option('flw_facebook_app_access_token', '');
							$__fb_updated = (int) get_option('flw_facebook_app_token_updated', 0);
							$__fb_expires = (int) get_option('flw_facebook_app_token_expires', 0);
							$__has_token  = $__fb_token !== '';
							$__fb_error   = get_transient('flw_fb_token_error');
						?>
						<div class="flw-token-status <?php echo $__has_token ? 'ok' : 'missing'; ?>">
							<strong><?php echo $__has_token ? 'App Access Token: Generated' : 'App Access Token: Not yet generated'; ?></strong>
							<?php if ($__has_token): ?>
								<span class="meta">Saved <?php echo esc_html( date_i18n( get_option('date_format') . ' ' . get_option('time_format'), $__fb_updated ?: time() ) ); ?><?php echo $__fb_expires ? ' · Expires ' . esc_html( date_i18n( get_option('date_format') . ' ' . get_option('time_format'), $__fb_expires ) ) : ' · No expiry reported'; ?></span>
							<?php elseif ($__fb_error): ?>
								<span class="meta" style="color:#d63638;">Last error: <?php echo esc_html( (string) $__fb_error ); ?></span>
							<?php endif; ?>
						</div>
						<label>App ID:
							<input type="text" name="flw_facebook_app_id" value="<?php echo esc_attr( get_option('flw_facebook_app_id', '') ); ?>" size="50" />
						</label>
						<br/>
						<label>App Secret:
							<input type="password" name="flw_facebook_app_secret" value="<?php echo esc_attr( get_option('flw_facebook_app_secret', '') ); ?>" size="50" />
						</label>
						<div class="flw-usage">
							<h3 style="margin:0 0 6px;">Using these Facebook settings in other plugins</h3>
							<p style="margin:0 0 8px;">On save, this page automatically generates an App Access Token using your App ID and Secret. Read values and initialize as needed:</p>
							<pre><code id="flw_facebook_usage_code">$appId      = get_option('flw_facebook_app_id', '');
$appSecret  = get_option('flw_facebook_app_secret', '');
$appToken   = get_option('flw_facebook_app_access_token', ''); // generated on save

// Example with Facebook PHP SDK (App token usage depends on endpoint):
// $fb = new \\Facebook\\Facebook([
//     'app_id' => $appId,
//     'app_secret' => $appSecret,
//     'default_graph_version' => 'v19.0',
// ]);
// $fb->setDefaultAccessToken($appToken);

// Optional: allow site-specific overrides via a filter in your plugin
$creds = [ 'app_id' => $appId, 'app_secret' => $appSecret, 'app_token' => $appToken ];
$creds = apply_filters('flw_facebook_credentials', $creds);
</code></pre>
							<button type="button" class="button" onclick="flw_copy_code_by_id('flw_facebook_usage_code', this)">Copy</button>
						</div>
						<div class="flw-usage">
							<h3 style="margin:0 0 6px;">How to create a Facebook App</h3>
							<ol style="margin:0; padding-left:18px;">
								<li>Go to <a href="https://developers.facebook.com/apps/" target="_blank" rel="noopener noreferrer">Meta for Developers — Apps</a> and click <strong>Create App</strong>.</li>
								<li>Select an app type that supports Graph API server-to-server calls (e.g., <strong>Business</strong>).</li>
								<li>Give it a name, complete setup, then open <strong>Settings → Basic</strong> to copy your <strong>App ID</strong> and <strong>App Secret</strong>.</li>
								<li>Ensure the app allows the <code>client_credentials</code> flow for obtaining an <strong>App Access Token</strong> (no user login required).</li>
								<li>Paste the App ID and Secret above and click <strong>Save Settings</strong>. We’ll auto-generate and store the App Access Token.</li>
								<li>If token generation fails, the banner will show the last error. Common causes:
									<ul>
										<li>Wrong app secret</li>
										<li>App not configured to allow client credentials</li>
										<li>Network or temporary API issue</li>
									</ul>
								</li>
							</ol>
						</div>
					</div>
					<div id="debug" class="tab-content">
						<h2>Debug Settings</h2>
						<label><input type="checkbox" name="flw_debug_mode" value="1" <?php checked(get_option('flw_debug_mode'),1); ?>/> Enable Debug Mode</label>

						<?php 
							$plugins = self::discover_flw_plugins();
							$toggles = (array) get_option('flw_plugin_debug_toggles', []);
						?>
						<h3 style="margin-top:16px;">Per‑Plugin Debugging</h3>
						<p class="description">Toggle debug logging for individual FLW plugins. Checked plugins may emit additional logs to <code>debug.log</code>.</p>
						<table class="widefat striped" style="max-width:760px; margin-top:8px;">
							<thead><tr><th style="width:46px;">On</th><th>Plugin</th><th>Status</th></tr></thead>
							<tbody>
							<?php if (empty($plugins)) : ?>
								<tr><td colspan="3">No FLW plugins detected. Other plugins can register via the <code>flw_debuggable_plugins</code> filter.</td></tr>
							<?php else: foreach ($plugins as $slug => $p): ?>
								<tr>
									<td><input type="checkbox" name="flw_plugin_debug_toggles[<?php echo esc_attr($slug); ?>]" value="1" <?php checked( !empty($toggles[$slug]) ); ?> /></td>
									<td><strong><?php echo esc_html($p['name']); ?></strong><br/><span style="color:#646970;"><?php echo esc_html($slug); ?></span></td>
									<td><?php echo $p['active'] ? '<span style="color:#2fb344;">Active</span>' : '<span style="color:#a7aaad;">Inactive</span>'; ?></td>
								</tr>
							<?php endforeach; endif; ?>
							</tbody>
						</table>

						<?php 
							$log_path = self::get_debug_log_path();
							$log_content = '';
							$log_exists = $log_path && file_exists($log_path);
							if ($log_exists && is_readable($log_path)) {
								$log_content = self::tail_file_lines($log_path, 3000);
							}
						?>
						<h3 style="margin:16px 0 6px;">debug.log (last 3000 lines)</h3>
					<div style="margin-bottom:8px; display:flex; gap:8px; align-items:center; flex-wrap: wrap;">
							<span class="description">Path: <code><?php echo esc_html( $log_path ?: 'Not configured (WP_DEBUG_LOG)'); ?></code></span>
							<button type="button" class="button" onclick="flw_copy_debug_log()">Copy</button>
						<button type="button" class="button button-secondary" onclick="flw_reset_debug_log(this)">Reset</button>
						</div>
						<textarea id="flw_debug_textarea" readonly style="width:100%; max-width:100%; height:320px; font-family:monospace; white-space:pre; overflow:auto;">
						<?php echo esc_textarea( $log_exists ? $log_content : "No log available. Ensure WP_DEBUG and WP_DEBUG_LOG are enabled." ); ?>
						</textarea>

						<div class="flw-usage">
							<h3 style="margin:0 0 6px;">Integrating plugin debug toggles</h3>
							<p style="margin:0 0 8px;">Register your plugin and read the toggle:</p>
							<pre><code id="flw_debug_usage_code">/** Declare your plugin in the Debug tab list */
add_filter('flw_debuggable_plugins', function(array $plugins): array {
    $plugins['my-plugin-slug'] = [
        'name' => 'My Plugin',
        // optional: 'active' => true/false (auto-detected if omitted)
    ];
    return $plugins;
});

/** Check if debug is enabled for your plugin */
$toggles = (array) get_option('flw_plugin_debug_toggles', []);
$isDebugEnabled = !empty($toggles['my-plugin-slug']);
if ($isDebugEnabled) {
    error_log('[My Plugin] debug message here');
}
</code></pre>
							<button type="button" class="button" onclick="flw_copy_code_by_id('flw_debug_usage_code', this)">Copy</button>
						</div>
					</div>
					<?php submit_button('Save Settings'); ?>
				</form>
			</div>
			<script>
				function showTab(e,id){e.preventDefault();document.querySelectorAll('.tab-content').forEach(c=>c.style.display='none');document.getElementById(id).style.display='block';document.querySelectorAll('.nav-tab').forEach(t=>t.classList.remove('nav-tab-active'));e.currentTarget.classList.add('nav-tab-active');}
				function flw_sync_stripe_groups(){
					var liveToggle = document.getElementById('flw_stripe_mode_live');
					var isLive = liveToggle && liveToggle.checked;
					var testGroup = document.getElementById('flw_stripe_keys_test');
					var liveGroup = document.getElementById('flw_stripe_keys_live');
					if(testGroup && liveGroup){
						testGroup.style.display = isLive ? 'none' : 'block';
						liveGroup.style.display = isLive ? 'block' : 'none';
					}
				}
				document.addEventListener('DOMContentLoaded', function(){
					var liveToggle = document.getElementById('flw_stripe_mode_live');
					if(liveToggle){
						liveToggle.addEventListener('change', flw_sync_stripe_groups);
					}
					// Open tab from URL hash if present; else default to Stripe
					var hash = window.location.hash || '#stripe';
					var link = document.querySelector('a.nav-tab[href="' + hash + '"]');
					if (!link) { link = document.querySelector('a.nav-tab[href="#stripe"]'); }
					if (link) { link.click(); }
					flw_sync_stripe_groups();
				});
				function flw_copy_debug_log(){
					var ta = document.getElementById('flw_debug_textarea');
					if(!ta){return;}
					if (navigator.clipboard && navigator.clipboard.writeText) {
						navigator.clipboard.writeText(ta.value);
					} else {
						ta.select(); document.execCommand('copy'); ta.selectionEnd = ta.selectionStart; // collapse selection
					}
				}
				function flw_copy_code_by_id(id, btn){
					var el = document.getElementById(id);
					if(!el){ return; }
					var text = el.innerText || el.textContent || '';
					if(!text){ return; }
					var original = btn ? btn.textContent : '';
					if (navigator.clipboard && navigator.clipboard.writeText) {
						navigator.clipboard.writeText(text).then(function(){ if(btn){ btn.textContent='Copied'; setTimeout(function(){ btn.textContent=original; }, 1200); } });
					} else {
						var ta = document.createElement('textarea');
						ta.value = text; document.body.appendChild(ta); ta.select(); document.execCommand('copy'); document.body.removeChild(ta);
						if(btn){ btn.textContent='Copied'; setTimeout(function(){ btn.textContent=original; }, 1200); }
					}
				}
				function flw_reset_debug_log(btn){
					if (!confirm('Clear debug.log? This cannot be undone.')) { return; }
					var url = "<?php echo esc_js( admin_url('admin-ajax.php') ); ?>";
					var nonce = "<?php echo esc_js( wp_create_nonce('flw_reset_debug_log') ); ?>";
					if (btn) { btn.disabled = true; }
					var body = 'action=flw_reset_debug_log&_ajax_nonce=' + encodeURIComponent(nonce);
					fetch(url, { method: 'POST', credentials: 'same-origin', headers: { 'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8' }, body })
						.then(function(r){ return r.json(); })
						.then(function(data){
							if (data && data.success) {
								var ta = document.getElementById('flw_debug_textarea'); if (ta) { ta.value = ''; }
							} else {
								alert((data && data.data && data.data.message) ? data.data.message : 'Failed to clear debug.log');
							}
						})
						.catch(function(){ alert('Failed to clear debug.log'); })
						.finally(function(){ if (btn) { btn.disabled = false; } });
				}
			</script>
			<style>
				.tab-content{display:none;padding:10px;border:1px solid #ccc;background:#fff;}
				.nav-tab-active{background:#fff;border-bottom-color:#fff;}
				.flw-toggle-row{display:flex;align-items:center;gap:10px;margin:8px 0 16px}
				.flw-switch{position:relative;display:inline-block;width:48px;height:24px}
				.flw-switch input{opacity:0;width:0;height:0}
				.flw-slider{position:absolute;cursor:pointer;top:0;left:0;right:0;bottom:0;background-color:#ccc;transition:.2s;border-radius:24px}
				.flw-slider:before{position:absolute;content:"";height:18px;width:18px;left:3px;bottom:3px;background-color:white;transition:.2s;border-radius:50%}
				.flw-switch input:checked + .flw-slider{background-color:#2271b1}
				.flw-switch input:checked + .flw-slider:before{transform:translateX(24px)}
				.flw-switch-label{vertical-align:middle}
				.flw-stripe-keys-group h3{margin:10px 0 6px}
					.flw-usage{margin-top:14px;padding:12px;background:#f8f9fb;border:1px solid #d0d7de;border-radius:6px}
					.flw-usage pre{background:#fff;padding:10px;border:1px solid #d0d7de;border-radius:6px;overflow:auto}
						.flw-token-status{margin:10px 0 14px;padding:10px;border-radius:6px;border:1px solid #d0d7de;background:#fff;display:flex;gap:10px;align-items:center}
						.flw-token-status.ok{border-color:#2fb344;background:#effaf3}
						.flw-token-status.missing{border-color:#d63638;background:#fff5f5}
						.flw-token-status .meta{color:#50575e}
			</style>
			<?php
		}

		/**
		 * Sanitize array of per-plugin debug toggles: keep checked slugs only.
		 */
		public static function sanitize_plugin_debug_toggles( $input ) {
			$sanitized = [];
			if ( is_array( $input ) ) {
				foreach ( $input as $slug => $val ) {
					$slug_key = sanitize_key( (string) $slug );
					$sanitized[ $slug_key ] = 1;
				}
			}
			return $sanitized;
		}

		/**
		 * Find FLW plugins in the installation and allow others to register via filter.
		 * @return array<string,array{name:string,active:bool}>
		 */
		private static function discover_flw_plugins(): array {
			if ( ! function_exists( 'get_plugins' ) ) {
				require_once ABSPATH . 'wp-admin/includes/plugin.php';
			}
			$all = get_plugins();
			$found = [];
			foreach ( $all as $file => $data ) {
				$name = isset( $data['Name'] ) ? (string) $data['Name'] : $file;
				$author = isset( $data['AuthorName'] ) ? (string) $data['AuthorName'] : ( (string) ( $data['Author'] ?? '' ) );
				$text_domain = isset( $data['TextDomain'] ) ? (string) $data['TextDomain'] : '';
				$slug_guess = $text_domain !== '' ? $text_domain : ( strpos( $file, '/' ) !== false ? dirname( $file ) : basename( $file, '.php' ) );
				$slug = sanitize_key( (string) $slug_guess );
				$is_flw = ( stripos( $author, 'FrontLine Works' ) !== false ) || preg_match( '/^flw[-_]/i', $slug ) || preg_match( '/^FLW\b/i', $name ) || preg_match( '/^flw[-_]/i', (string) $text_domain );
				if ( ! $is_flw ) { continue; }
				$active = function_exists('is_plugin_active') ? is_plugin_active( $file ) : in_array( $file, (array) get_option( 'active_plugins', [] ), true );
				$found[ $slug ] = [ 'name' => $name, 'active' => (bool) $active ];
			}
			/** Allow other plugins to register or modify debuggable plugin entries */
			$found = apply_filters( 'flw_debuggable_plugins', $found );
			// Normalize shape to expected fields
			foreach ( $found as $slug => $p ) {
				$found[ $slug ] = [
					'name'   => isset( $p['name'] ) ? (string) $p['name'] : ( is_string( $p ) ? $p : (string) $slug ),
					'active' => isset( $p['active'] ) ? (bool) $p['active'] : true,
				];
			}
			// Sort by name for stable display
			uasort( $found, function( $a, $b ){ return strcasecmp( (string)$a['name'], (string)$b['name'] ); } );
			return $found;
		}

		/**
		 * Resolve the debug.log file path, respecting WP_DEBUG_LOG custom paths.
		 */
		private static function get_debug_log_path(): string {
			$path = '';
			if ( defined( 'WP_DEBUG_LOG' ) ) {
				if ( WP_DEBUG_LOG === true ) {
					$path = trailingslashit( WP_CONTENT_DIR ) . 'debug.log';
				} elseif ( is_string( WP_DEBUG_LOG ) && WP_DEBUG_LOG !== '' ) {
					$path = WP_DEBUG_LOG;
				}
			}
			if ( $path === '' ) {
				$path = trailingslashit( WP_CONTENT_DIR ) . 'debug.log';
			}
			return $path;
		}

		/**
		 * Return the last N lines from a file efficiently.
		 */
		private static function tail_file_lines( string $file, int $lines ): string {
			$lines = max(1, (int)$lines);
			$fp = @fopen( $file, 'rb' );
			if ( ! $fp ) { return ''; }
			$buffer = '';
			$chunkSize = 8192;
			fseek( $fp, 0, SEEK_END );
			$pos = ftell( $fp );
			$lineCount = 0;
			while ( $pos > 0 && $lineCount <= $lines ) {
				$read = ( $pos - $chunkSize ) >= 0 ? $chunkSize : $pos;
				$pos -= $read;
				fseek( $fp, $pos, SEEK_SET );
				$chunk = fread( $fp, $read );
				$buffer = $chunk . $buffer;
				$lineCount = substr_count( $buffer, "\n" );
			}
			fclose( $fp );
			$parts = explode( "\n", $buffer );
			$parts = array_slice( $parts, -$lines );
			return implode( "\n", $parts );
		}

		/**
		 * Handle POST to clear the debug.log file.
		 */
		public static function handle_reset_debug_log() {
			if ( ! current_user_can( 'manage_options' ) ) { wp_die( 'Unauthorized' ); }
			check_admin_referer( 'flw_reset_debug_log' );
			$path = self::get_debug_log_path();
			$ok = true;
			if ( $path && file_exists( $path ) ) {
				$ok = is_writable( $path ) ? ( file_put_contents( $path, '' ) !== false ) : false;
			} else {
				// If file doesn't exist, consider it cleared
				$ok = true;
			}
			$dest = admin_url( 'admin.php?page=flw-global-settings#debug' );
			$dest = add_query_arg( 'flw_debug_reset', $ok ? '1' : '0', $dest );
			wp_safe_redirect( $dest );
			exit;
		}

		/**
		 * AJAX handler to clear the debug.log without redirect
		 */
		public static function ajax_reset_debug_log() {
			if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( [ 'message' => 'Unauthorized' ], 403 ); }
			check_ajax_referer( 'flw_reset_debug_log' );
			$path = self::get_debug_log_path();
			$ok = true;
			if ( $path && file_exists( $path ) ) {
				$ok = is_writable( $path ) ? ( file_put_contents( $path, '' ) !== false ) : false;
			}
			if ( $ok ) { wp_send_json_success( [ 'cleared' => true ] ); }
			wp_send_json_error( [ 'message' => 'debug.log not writable' ], 500 );
		}
	}
	FLW_Plugin_Library::initialize();
}

