<?php
if (!defined('ABSPATH')) { exit; }

require_once __DIR__ . '/rest/request.php';

/**
 * Check if this site is entitled to auto-updates (has active subscription).
 * Uses a short-lived transient cache to avoid frequent remote calls.
 *
 * @return bool
 */
function flw_su_is_entitled_for_updates(): bool {
	$cached = get_transient('flw_su_entitlement_status');
	if (is_array($cached) && isset($cached['ok']) && is_bool($cached['ok'])) {
		return (bool)$cached['ok'];
	}

	$client_id = get_option(defined('FLW_SU_OPTION_CLIENT_ID') ? FLW_SU_OPTION_CLIENT_ID : 'flw_su_client_id', '');
	if ($client_id === '') {
		set_transient('flw_su_entitlement_status', ['ok' => false], MINUTE_IN_SECONDS * 5);
		return false;
	}

	$result = flw_su_remote_request('GET', '/flw_su/v1/subscriptions/status');
	$ok = false;
	if (is_array($result) && (int)($result['code'] ?? 500) === 200 && is_array($result['json'] ?? null)) {
		$json = $result['json'];
		$active = !empty($json['active']);
		$exempt = !empty($json['exempt']);
		$ok = ($active || $exempt);
	}
	// Cache success a bit longer; failures shorter.
	set_transient('flw_su_entitlement_status', ['ok' => $ok], $ok ? MINUTE_IN_SECONDS * 15 : MINUTE_IN_SECONDS * 5);
	return $ok;
}

/**
 * Message displayed under plugin row when updates are gated.
 *
 * @return string
 */
function flw_su_get_non_entitled_message(): string {
	return esc_html__('An update is available. Auto-update requires an active FLW subscription. Visit the FLW website to download the update.', 'flw-secure-updates-local');
}

/**
 * Clear entitlement cache. Call after subscription changes.
 */
function flw_su_clear_entitlement_cache(): void {
	delete_transient('flw_su_entitlement_status');
}

