<?php
if (!defined('ABSPATH')) { exit; }

/**
 * Track available updates per plugin file for dashboard and notices.
 * Stores a transient map: [ plugin_basename => bool has_update ]
 */
function flw_su_set_update_available(string $pluginFile, bool $has): void {
	$map = get_transient('flw_su_update_map');
	if (!is_array($map)) { $map = []; }
	$map[$pluginFile] = $has ? 1 : 0;
	// Keep the cache relatively fresh to avoid stale banners.
	set_transient('flw_su_update_map', $map, 15 * MINUTE_IN_SECONDS);
}

/**
 * Get map of updates detected by our system.
 * @return array<string,int>
 */
function flw_su_get_update_map(): array {
	$map = get_transient('flw_su_update_map');
	return is_array($map) ? $map : [];
}

/**
 * Whether any monitored plugin has an update (for admin banner).
 */
function flw_su_any_updates_available(): bool {
	$map = flw_su_get_update_map();
	foreach ($map as $has) { if (!empty($has)) return true; }
	return false;
}

/**
 * Clear update flags for specific plugin files (after successful upgrade).
 *
 * @param string[] $pluginFiles
 */
function flw_su_clear_update_flags(array $pluginFiles): void {
	if (empty($pluginFiles)) { return; }
	$map = get_transient('flw_su_update_map');
	if (!is_array($map)) { return; }
	foreach ($pluginFiles as $file) {
		$map[$file] = 0;
	}
	set_transient('flw_su_update_map', $map, 15 * MINUTE_IN_SECONDS);
}

/**
 * Prune update flags for plugins that are no longer installed.
 */
function flw_su_prune_update_map(): void {
	$map = get_transient('flw_su_update_map');
	if (!is_array($map)) { return; }
	if (!function_exists('get_plugins')) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
	}
	$installed = array_keys(get_plugins());
	$installed = is_array($installed) ? $installed : [];
	$installedFlip = array_fill_keys($installed, 1);
	$changed = false;
	foreach (array_keys($map) as $file) {
		if (!isset($installedFlip[$file])) { unset($map[$file]); $changed = true; }
	}
	if ($changed) {
		set_transient('flw_su_update_map', $map, 15 * MINUTE_IN_SECONDS);
	}
}

