<?php
/**
 * FLW Secure Updates - Global Multi-Plugin Update Manager
 *
 * Scans installed plugins for specific authors and wires Plugin Update Checker (PUC)
 * without requiring code inside those plugins.
 */

if (!defined('ABSPATH')) {
	exit;
}

/**
 * Derive an update slug for a plugin.
 *
 * Prefers plugin folder name. Falls back to TextDomain or main file name.
 *
 * @param string $pluginFile Relative plugin file path from plugins dir (e.g. my-plugin/my-plugin.php)
 * @param array  $pluginData Header data from get_plugins()[ $pluginFile ]
 * @return string Slug candidate used for metadata URL
 */
function flw_su_derive_slug_from_plugin_file($pluginFile, $pluginData) {
	$directoryName = trim(basename(dirname($pluginFile)));
	if ($directoryName !== '' && $directoryName !== '.' && $directoryName !== '/') {
		return sanitize_key($directoryName);
	}
	if (!empty($pluginData['TextDomain'])) {
		return sanitize_key($pluginData['TextDomain']);
	}
	$baseName = basename($pluginFile, '.php');
	return sanitize_key($baseName);
}

/**
 * Initialize update checking for plugins authored by the allowed authors list.
 */
function flw_su_initialize_global_update_manager() {
	// Only run in admin or cron to avoid front-end overhead.
	if (!is_admin() && !wp_doing_cron()) {
		return;
	}

	// Ensure dependencies.
	if (!function_exists('get_plugins')) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
	}

	if (!class_exists('FLW_SU_Update_Checker')) {
		return;
	}

	static $initializedBasenames = null;
	if ($initializedBasenames === null) {
		$initializedBasenames = [];
	}

	$defaultAuthors = [
		'Tyson Brooks',
		'FrostLine Works',
		'FrontLine Works',
	];
	$allowedAuthors = apply_filters('flw_su_allowed_authors', $defaultAuthors);
	$base = get_option('flw_metadata_url', 'https://frostlineworks.com/updates/');

	$allPlugins = get_plugins();
	if (empty($allPlugins)) {
		return;
	}

	foreach ($allPlugins as $pluginFile => $pluginData) {
		$author = isset($pluginData['Author']) ? wp_strip_all_tags((string)$pluginData['Author']) : '';
		$authorName = isset($pluginData['AuthorName']) ? wp_strip_all_tags((string)$pluginData['AuthorName']) : '';
		$haystack = $author . ' ' . $authorName;

		$matchesAuthor = false;
		foreach ((array)$allowedAuthors as $allowed) {
			if ($allowed !== '' && stripos($haystack, (string)$allowed) !== false) {
				$matchesAuthor = true;
				break;
			}
		}

		if (!$matchesAuthor) {
			continue;
		}

		$absolutePluginPath = trailingslashit(WP_PLUGIN_DIR) . $pluginFile;
		$basename = plugin_basename($absolutePluginPath);

		// Avoid double-initialization.
		if (isset($initializedBasenames[$basename])) {
			continue;
		}


		$slug = flw_su_derive_slug_from_plugin_file($pluginFile, $pluginData);
		// For this plugin (Local), override to repository slug if needed
		$isSelf = (plugin_basename($absolutePluginPath) === plugin_basename(FLW_SU_LOCAL_PATH . 'local.php'));
		if ($isSelf) {
			$slug = apply_filters('flw_su_self_update_slug', 'flwsecureupdates');
		}

		// Quick metadata preflight to avoid creating checkers for unrelated slugs
		$preflightUrl = add_query_arg([
			'action' => 'get_metadata',
			'slug'   => $slug,
		], $base);
		$resp = wp_remote_get($preflightUrl, ['timeout' => 5]);
		if (is_wp_error($resp) || wp_remote_retrieve_response_code($resp) !== 200) {
			$initializedBasenames[$basename] = 'meta-missing';
			continue;
		}

		$checker = FLW_SU_Update_Checker::initialize($absolutePluginPath, $slug);

		// Track updates availability and hide non-entitled updates from WP updater UI
		if ( $checker ) {
			$checker->addFilter('request_update_result', function($update, $httpResult = null) use ($pluginFile, $absolutePluginPath, $pluginData) {
				$basename = plugin_basename($absolutePluginPath);
				$isSelf = ($basename === plugin_basename(FLW_SU_LOCAL_PATH . 'local.php'));
				if ($update) {
					// Only consider it an update if remote version > installed version
					$installed = (string)($pluginData['Version'] ?? '0');
					$remote = (string)($update->version ?? '0');
					if (version_compare($remote, $installed, '<=')) {
						if (function_exists('flw_su_set_update_available')) { flw_su_set_update_available($basename, false); }
						return null;
					}
					// Mark as available in our map
					if (function_exists('flw_su_set_update_available')) { flw_su_set_update_available($basename, true); }
					// If not entitled and not self, suppress it from WP updater by returning null
					if (!$isSelf && function_exists('flw_su_is_entitled_for_updates') && !flw_su_is_entitled_for_updates()) {
						return null;
					}
				} else {
					if (function_exists('flw_su_set_update_available')) { flw_su_set_update_available($basename, false); }
				}
				return $update;
			}, 10, 2);
		}

		// Mark as initialized whether or not metadata was found to prevent repeated attempts during this request.
		$initializedBasenames[$basename] = $checker ? 'ok' : 'meta-missing';
	}
}

// Run late on init so core and plugins are loaded; still early enough for update hooks.
add_action('init', 'flw_su_initialize_global_update_manager', 15);

