<?php
/**
Plugin Name: FLW Secure Updates - Local
Description: Local client plugin that registers with Remote and provides subscription UI.
Version: 1.0.23
Requires at least: 6.5
Requires PHP: 8.0
Author: FrontLine Works
Text Domain: flw-secure-updates-local
*/

if (!defined('ABSPATH')) {
	exit;
}

// Define constants
if (!defined('FLW_SU_LOCAL_VERSION')) {
    define('FLW_SU_LOCAL_VERSION', '1.0.23');
}
if (!defined('FLW_SU_LOCAL_PATH')) {
	define('FLW_SU_LOCAL_PATH', plugin_dir_path(__FILE__));
}
if (!defined('FLW_SU_LOCAL_URL')) {
	define('FLW_SU_LOCAL_URL', plugin_dir_url(__FILE__));
}
if (!defined('FLW_SU_BROKER_BASE')) {
	define('FLW_SU_BROKER_BASE', 'https://frostlineworks.com');
}

// Options keys
if (!defined('FLW_SU_OPTION_CLIENT_ID')) {
	define('FLW_SU_OPTION_CLIENT_ID', 'flw_su_client_id');
}
if (!defined('FLW_SU_OPTION_CLIENT_SECRET')) {
	define('FLW_SU_OPTION_CLIENT_SECRET', 'flw_su_client_secret');
}
if (!defined('FLW_SU_OPTION_BROKER_URL')) {
	define('FLW_SU_OPTION_BROKER_URL', 'flw_su_broker_base_url');
}

// Autoload includes (manual lightweight loader)
require_once FLW_SU_LOCAL_PATH . 'includes/admin/settings-page.php';
require_once FLW_SU_LOCAL_PATH . 'includes/admin/dashboard-page.php';
require_once FLW_SU_LOCAL_PATH . 'includes/admin/ajax.php';
require_once FLW_SU_LOCAL_PATH . 'includes/admin/library-menu.php';
require_once FLW_SU_LOCAL_PATH . 'includes/rest/hmac-signer.php';
require_once FLW_SU_LOCAL_PATH . 'includes/rest/request.php';
require_once FLW_SU_LOCAL_PATH . 'includes/subscription.php';
require_once FLW_SU_LOCAL_PATH . 'includes/update-status.php';
require_once FLW_SU_LOCAL_PATH . 'includes/update-manager.php';

// Activation hook
register_activation_hook(__FILE__, function () {
	// Ensure options exist
	add_option(FLW_SU_OPTION_CLIENT_ID, '');
	add_option(FLW_SU_OPTION_CLIENT_SECRET, '');
	add_option(FLW_SU_OPTION_BROKER_URL, FLW_SU_BROKER_BASE);

	// Deactivate legacy FLW Plugin Library if present and active
	if (!function_exists('is_plugin_active')) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
	}
	$legacy = 'flwpluginlibrary/flwpluginlibrary.php';
	if (function_exists('is_plugin_active')) {
		$network_active = function_exists('is_plugin_active_for_network') && is_plugin_active_for_network($legacy);
		if (is_plugin_active($legacy) || $network_active) {
			deactivate_plugins($legacy, true, $network_active);
			update_option('flw_su_legacy_deactivated', 1);
		}
	}
});

// Register settings
add_action('admin_init', function () {
	register_setting('flw_su_local', FLW_SU_OPTION_BROKER_URL, [
		'type' => 'string',
		'default' => FLW_SU_BROKER_BASE,
	]);
});

// Admin menu
add_action('admin_menu', function () {
	if (!current_user_can('manage_options')) {
		return;
	}
	add_menu_page(
		__('Secure Updates', 'flw-secure-updates-local'),
		__('Secure Updates', 'flw-secure-updates-local'),
		'manage_options',
		'flw_su_dashboard',
		'flw_su_render_dashboard_page',
		'dashicons-shield-alt'
	);
	add_submenu_page(
		'flw_su_dashboard',
		__('Dashboard', 'flw-secure-updates-local'),
		__('Dashboard', 'flw-secure-updates-local'),
		'manage_options',
		'flw_su_dashboard',
		'flw_su_render_dashboard_page',
		0
	);
	add_submenu_page(
		'flw_su_dashboard',
		__('Settings', 'flw-secure-updates-local'),
		__('Settings', 'flw-secure-updates-local'),
		'manage_options',
		'flw_su_settings',
		'flw_su_render_settings_page',
		1
	);
});

// Enqueue admin assets
add_action('admin_enqueue_scripts', function ($hook) {
	$page = isset($_GET['page']) ? sanitize_key($_GET['page']) : '';
	if ($page !== 'flw_su_dashboard' && $page !== 'flw_su_settings') {
		return;
	}
	wp_enqueue_script('flw-su-admin', FLW_SU_LOCAL_URL . 'assets/admin.js', [], FLW_SU_LOCAL_VERSION, true);
	wp_localize_script('flw-su-admin', 'FLW_SU', [
		'ajax_url' => admin_url('admin-ajax.php'),
		'nonce' => wp_create_nonce('flw_su_admin'),
	]);
});

// Global admin banner when updates are available
add_action('admin_notices', function(){
	if (!current_user_can('update_plugins')) { return; }
	if (!function_exists('flw_su_any_updates_available')) { return; }
	$user_id = get_current_user_id();
	if ($user_id) {
		$until = (int) get_user_meta($user_id, 'flw_su_dismiss_updates_notice_until', true);
		if ($until > time()) { return; }
	}
	if (function_exists('flw_su_prune_update_map')) { flw_su_prune_update_map(); }
	$has = flw_su_any_updates_available();
	if (!$has) { return; }
	// If entitled, link to WP updates; else link to our dashboard
	$targetUrl = function_exists('flw_su_is_entitled_for_updates') && flw_su_is_entitled_for_updates()
		? admin_url('update-core.php')
		: admin_url('admin.php?page=flw_su_dashboard');
	$dismiss_url = wp_nonce_url(
		admin_url('admin-post.php?action=flw_su_dismiss_updates_notice&redirect_to=' . rawurlencode(isset($_SERVER['REQUEST_URI']) ? (string)$_SERVER['REQUEST_URI'] : admin_url('index.php'))),
		'flw_su_dismiss_updates_notice'
	);
	echo '<div class="notice notice-error is-dismissible" style="border-left-color:#d63638"><p>'
		. esc_html__('FLW: One or more plugins have updates available.', 'flw-secure-updates-local')
		. ' <a href="' . esc_url($targetUrl) . '" class="button button-small" style="margin-left:6px;">'
		. esc_html__('View Updates', 'flw-secure-updates-local')
		. '</a>'
		. ' <a href="' . esc_url($dismiss_url) . '" class="button button-small" style="margin-left:6px;">'
		. esc_html__('Dismiss for 24 hours', 'flw-secure-updates-local')
		. '</a>'
		. '</p></div>';
});

// On admin load, ensure legacy plugin is deactivated if still active
add_action('admin_init', function(){
	if (!current_user_can('activate_plugins')) { return; }
	if (!function_exists('is_plugin_active')) {
		require_once ABSPATH . 'wp-admin/includes/plugin.php';
	}
	$legacy = 'flwpluginlibrary/flwpluginlibrary.php';
	$network_active = function_exists('is_plugin_active_for_network') && is_plugin_active_for_network($legacy);
	if (function_exists('is_plugin_active') && (is_plugin_active($legacy) || $network_active)) {
		deactivate_plugins($legacy, true, $network_active);
		update_option('flw_su_legacy_deactivated', 1);
	}
});

// One-time admin notice if we deactivated the legacy plugin
add_action('admin_notices', function(){
	if (!current_user_can('activate_plugins')) { return; }
	if ((int) get_option('flw_su_legacy_deactivated', 0) !== 1) { return; }
	delete_option('flw_su_legacy_deactivated');
	echo '<div class="notice notice-warning is-dismissible"><p>'
		. esc_html__('FLW Secure Updates replaced and deactivated the old "FLW Plugin Library" plugin.', 'flw-secure-updates-local')
		. '</p></div>';
});

// Handler to dismiss the updates banner for 24 hours (per-user)
function flw_su_dismiss_updates_notice_handler() {
	if ( ! current_user_can( 'update_plugins' ) ) { wp_die( 'Unauthorized user' ); }
	check_admin_referer('flw_su_dismiss_updates_notice');
	$user_id = get_current_user_id();
	if ($user_id) {
		update_user_meta($user_id, 'flw_su_dismiss_updates_notice_until', time() + DAY_IN_SECONDS);
	}
	$redirect = isset( $_GET['redirect_to'] ) ? esc_url_raw( $_GET['redirect_to'] ) : admin_url( 'plugins.php' );
	wp_safe_redirect( $redirect );
	exit;
}
add_action( 'admin_post_flw_su_dismiss_updates_notice', 'flw_su_dismiss_updates_notice_handler' );

// When plugins are updated/installed, clear our internal update flags and refresh checks
add_action('upgrader_process_complete', function($upgrader, $hook_extra){
	if (!is_array($hook_extra)) { return; }
	if (($hook_extra['type'] ?? '') !== 'plugin') { return; }
	$action = isset($hook_extra['action']) ? (string)$hook_extra['action'] : '';
	if ($action !== 'update' && $action !== 'upgrade' && $action !== 'install') { return; }
	$files = [];
	if (!empty($hook_extra['plugins']) && is_array($hook_extra['plugins'])) {
		$files = $hook_extra['plugins'];
	} elseif (!empty($hook_extra['plugin']) && is_string($hook_extra['plugin'])) {
		$files = [ $hook_extra['plugin'] ];
	}
	if (!empty($files) && function_exists('flw_su_clear_update_flags')) {
		flw_su_clear_update_flags($files);
	}
	if (function_exists('wp_clean_plugins_cache')) { wp_clean_plugins_cache(true); }
	if (function_exists('wp_update_plugins')) { wp_update_plugins(); }
}, 10, 2);

// Also prune flags when the Updates screen is loaded/refreshed
add_action('load-update-core.php', function(){
	if (!current_user_can('update_plugins')) { return; }
	if (function_exists('flw_su_prune_update_map')) { flw_su_prune_update_map(); }
}, 20);

/* =======================================
   FLW Plugin Update Checker Class
   ======================================= */
   if ( ! class_exists( 'FLW_SU_Update_Checker' ) ) {
    class FLW_SU_Update_Checker {
        public static function initialize( $pluginFile, $slug ) {
            require_once plugin_dir_path( __FILE__ ) . 'plugin-update-checker/plugin-update-checker.php';

            // Missing library?
            if ( ! class_exists( 'FLWPlugins\PluginUpdateChecker\v5\PucFactory' ) ) {
                if ( ! get_transient( 'flw_update_checker_lib_missing' ) ) {
                    error_log( 'FLW Plugin Library: Update Checker library is missing.' );
                    set_transient( 'flw_update_checker_lib_missing', 1, HOUR_IN_SECONDS );
                }
                return null;
            }

            // Fetch metadata
            $base = get_option( 'flw_metadata_url', 'https://frostlineworks.com/updates/' );
            $url  = add_query_arg( [ 'action' => 'get_metadata', 'slug' => $slug ], $base );
            $resp = wp_remote_get( $url, [ 'timeout' => 5 ] );

            if ( ! is_wp_error( $resp ) && wp_remote_retrieve_response_code( $resp ) === 200 ) {
                // Validate that the metadata endpoint returns valid JSON before initializing PUC.
                $body = wp_remote_retrieve_body( $resp );
                $decoded = json_decode( $body );
                if ( empty( $decoded ) || ! is_object( $decoded ) ) {
                    $mkey = 'flw_update_checker_invalid_json_' . $slug;
                    if ( ! get_transient( $mkey ) ) {
                        error_log( "FLW Plugin Library: Invalid JSON received from {$url}. Skipping update checker init for {$slug}." );
                        set_transient( $mkey, 1, HOUR_IN_SECONDS );
                    }
                    return null;
                }
                $checker = \FLWPlugins\PluginUpdateChecker\v5\PucFactory::buildUpdateChecker(
                    $url,
                    $pluginFile,
                    $slug
                );
                // Gate auto-update package URLs behind subscription entitlement for third-party plugins (not this plugin)
                $is_self_plugin = plugin_basename( $pluginFile ) === plugin_basename( __FILE__ );
                if ( ! $is_self_plugin && function_exists('flw_su_is_entitled_for_updates') ) {
                    $checker->addFilter( 'pre_inject_update', function( $update ) {
                        if ( $update && ! flw_su_is_entitled_for_updates() ) {
                            $update->download_url = '';
                        }
                        return $update;
                    } );
                    $checker->addFilter( 'request_info_result', function( $info ) {
                        if ( $info && ! flw_su_is_entitled_for_updates() ) {
                            $info->download_url = '';
                        }
                        return $info;
                    } );
                    // Show an informational message in the plugin update row when not entitled
                    add_action( 'in_plugin_update_message_' . plugin_basename( $pluginFile ), function( $plugin_data, $response ) {
                        if ( ! flw_su_is_entitled_for_updates() ) {
                            $message = function_exists('flw_su_get_non_entitled_message') ? flw_su_get_non_entitled_message() : __( 'An update is available. Auto-update requires an active FLW subscription.', 'flw-secure-updates-local' );
                            echo '<br /><span class="flw-su-update-note">' . esc_html( $message ) . '</span>';
                        }
                    }, 10, 2 );
                }
                // (Removed custom action-link to avoid duplicate "Check for updates" alongside PUC link)

                // Add plugin icons to the update response (matches gist behavior)
                add_filter( 'site_transient_update_plugins', function( $transient ) use ( $pluginFile ) {
                    if ( empty( $transient ) || empty( $transient->response ) || ! is_array( $transient->response ) ) {
                        return $transient;
                    }
                    $basename = plugin_basename( $pluginFile );
                    if ( isset( $transient->response[ $basename ] ) ) {
                        $icon128 = plugins_url( 'assets/logo-128x128.png', $pluginFile );
                        $icon256 = plugins_url( 'assets/logo-256x256.png', $pluginFile );
                        if ( ! isset( $transient->response[ $basename ]->icons ) || ! is_array( $transient->response[ $basename ]->icons ) ) {
                            $transient->response[ $basename ]->icons = [];
                        }
                        $transient->response[ $basename ]->icons['default'] = $icon128;
                        $transient->response[ $basename ]->icons['1x']      = $icon128;
                        $transient->response[ $basename ]->icons['2x']      = $icon256;
                    }
                    return $transient;
                }, 10, 1 );
                return $checker;
            }

            // Metadata error
            $mkey = 'flw_update_checker_meta_error_' . $slug;
            if ( ! get_transient( $mkey ) ) {
                error_log( "FLW Plugin Library: Metadata file not found at {$url}. Update checker not initialized for {$slug}." );
                set_transient( $mkey, 1, HOUR_IN_SECONDS );
            }
            return null;
        }

        public function replace_plugin_update_icon( $transient ) {
            if ( ! empty( $transient->response ) ) {
                foreach ( $transient->response as $slug => $data ) {
                    if ( $slug === plugin_basename( $data->plugin ) ) {
                        $icon = plugins_url( 'assets/logo-128x128.png', $data->plugin );
                        $transient->response[ $slug ]->icons = [
                            'default' => $icon,
                            '1x'      => $icon,
                            '2x'      => plugins_url( 'assets/logo-256x256.png', $data->plugin ),
                        ];
                        break;
                    }
                }
            }
            return $transient;
        }
    }
}

// Replace update icons for all FLW plugins by author match (global)
add_filter('site_transient_update_plugins', function($transient){
    if ( empty($transient) || empty($transient->response) || !is_array($transient->response) ) { return $transient; }

    if ( !function_exists('get_plugins') ) { require_once ABSPATH . 'wp-admin/includes/plugin.php'; }
    $plugins = function_exists('get_plugins') ? get_plugins() : [];
    $own = [];
    foreach ($plugins as $basename => $headers) {
        $author = isset($headers['Author']) ? wp_strip_all_tags($headers['Author']) : '';
        if (
            stripos($author, 'Tyson Brooks') !== false ||
            stripos($author, 'FrostLine Works') !== false ||
            stripos($author, 'FrontLine Works') !== false
        ) {
            $own[$basename] = true;
        }
    }

    if ( empty($own) ) { return $transient; }

    foreach ($transient->response as $basename => $data) {
        if ( isset($own[$basename]) ) {
            // Resolve icons relative to the plugin's own directory
            $pluginFile = isset($data->plugin) && is_string($data->plugin) ? $data->plugin : $basename;
            $absPluginFile = trailingslashit(WP_PLUGIN_DIR) . ltrim($pluginFile, '/');
            $icon128 = plugins_url('assets/logo-128x128.png', $absPluginFile);
            $icon256 = plugins_url('assets/logo-256x256.png', $absPluginFile);

            if ( !isset($transient->response[$basename]->icons) || !is_array($transient->response[$basename]->icons) ) {
                $transient->response[$basename]->icons = [];
            }
            $transient->response[$basename]->icons['default'] = $icon128;
            $transient->response[$basename]->icons['1x'] = $icon128;
            $transient->response[$basename]->icons['2x'] = $icon256;
        }
    }

    return $transient;
}, 5, 1);

/* =======================================
   Force Update Check (Admin Action) + Action Link
   ======================================= */
function flw_su_force_update_check_handler() {
    if ( ! current_user_can( 'update_plugins' ) ) {
        wp_die( 'Unauthorized user' );
    }
    check_admin_referer( 'flw_force_update_check' );

    // Clear caches and force a fresh check
    if ( function_exists( 'wp_clean_plugins_cache' ) ) {
        wp_clean_plugins_cache( true );
    } else {
        delete_site_transient( 'update_plugins' );
    }

    if ( function_exists( 'wp_update_plugins' ) ) {
        // Triggers an immediate update check
        wp_update_plugins();
    }

    $redirect = isset( $_GET['redirect_to'] ) ? esc_url_raw( $_GET['redirect_to'] ) : admin_url( 'plugins.php' );
    wp_safe_redirect( add_query_arg( 'flw_updates_checked', '1', $redirect ) );
    exit;
}
add_action( 'admin_post_flw_force_update_check', 'flw_su_force_update_check_handler' );

// Add a "Check for updates" link for the FLW Plugin Library itself
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), function( $links ) {
    $force_url = admin_url( 'admin-post.php?action=flw_force_update_check&_wpnonce=' . wp_create_nonce( 'flw_force_update_check' ) . '&redirect_to=' . rawurlencode( admin_url( 'plugins.php' ) ) );
    $links[]   = '<a href="' . esc_url( $force_url ) . '">' . esc_html__( 'Check for updates', 'flw-plugin-library' ) . '</a>';
    return $links;
} );
