<?php
/*
Plugin Name: Patreon User Role Changer
Description: Changes the role of newly created users with a username starting with "patreon_" to a configurable role.
Version: 1.2.0
Author: FrostLine Works
Author URI: https://frostlineworks.com
Tested up to: 6.3
Requires Plugins: flwsecureupdates
*/

// Prevent direct access to this file
if (!defined('ABSPATH')) {
    exit;
}

// Add a "Settings" link to the plugin actions in the plugins list
add_filter('plugin_action_links_' . plugin_basename(__FILE__), function ($links) {
    $pluginSlug = basename(dirname(__FILE__));
    $settings_link = '<a href="admin.php?page=' . $pluginSlug . '">Settings</a>';
    array_unshift($links, $settings_link);
    return $links;
});

// Ensure the FLW Plugin Library is loaded before running the plugin
add_action('plugins_loaded', function () {

        class Patreon_User_Role_Changer_Plugin {

            private $option_name = 'patreon_user_role';

            public function __construct() {
                add_action('user_register', [$this, 'change_user_role'], 10, 1);
                add_action('admin_menu', [$this, 'register_submenu']);
                add_action('admin_init', [$this, 'register_settings']);
            }

            /**
             * Changes the role of users based on their Patreon membership tier.
             *
             * @param int $user_id The ID of the newly registered user.
             */
            public function change_user_role($user_id) {
                $user = get_userdata($user_id);
                if (!$user) {
                    return;
                }

                // Check if the username starts with 'patreon_'
                if (strpos($user->user_login, 'patreon_') === 0) {
                    $patreon_id = str_replace('patreon_', '', $user->user_login); // Extract Patreon ID from username
                    $pledge_level = $this->get_patreon_pledge_level($patreon_id);

                    // Define pledge level to WordPress role mapping
                    $pledge_roles = [
                        'tier_1_id' => 'subscriber',
                        'tier_2_id' => 'contributor',
                        'tier_3_id' => 'author',
                        'tier_4_id' => 'editor',
                    ];

                    if ($pledge_level && isset($pledge_roles[$pledge_level])) {
                        $user->set_role($pledge_roles[$pledge_level]);
                    } else {
                        $user->set_role(get_option($this->option_name, 'subscriber')); // Default role if pledge level not found
                    }
                }
            }


            /**
             * Register the submenu under the FLW Plugins menu.
             */
            public function register_submenu() {
                $pluginSlug = basename(dirname(__FILE__));
                FLW_Plugin_Library::add_submenu(
                    'Patreon User Role',
                    $pluginSlug,
                    [$this, 'render_settings_page']
                );
            }

            /**
             * Registers the plugin settings.
             */
            public function register_settings() {
                register_setting('patreon_user_role_settings', $this->option_name, [
                    'type' => 'string',
                    'sanitize_callback' => [$this, 'sanitize_role'],
                    'default' => 'subscriber',
                ]);
                // Store Patreon API Token
                register_setting('patreon_user_role_settings', 'patreon_api_token', [
                    'type' => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                ]);

            }

            /**
             * Sanitizes the role input.
             *
             * @param string $role The role selected by the admin.
             * @return string Sanitized role.
             */
            public function sanitize_role($role) {
                return get_role($role) ? $role : 'subscriber';
            }

            /**
             * Render the settings page content.
             */
            public function render_settings_page() {
                global $wp_roles;
                $roles = $wp_roles->roles;
                $selected_role = get_option($this->option_name, 'subscriber');
                ?>
                <div class="wrap">
                    <h1>Patreon User Role Settings</h1>
                    <form method="post" action="options.php">
                        <?php
                        settings_fields('patreon_user_role_settings');
                        do_settings_sections('patreon_user_role_settings');
                        ?>
                        <table class="form-table">
                            <tr valign="top">
                                <th scope="row">Role for Patreon Members</th>
                                <td>
                                    <select name="<?php echo esc_attr($this->option_name); ?>">
                                        <?php foreach ($roles as $role_key => $role_data) : ?>
                                            <option value="<?php echo esc_attr($role_key); ?>" <?php selected($selected_role, $role_key); ?>>
                                                <?php echo esc_html($role_data['name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                    <p class="description">Select the role to assign to new Patreon users.</p>
                                </td>
                            </tr>
                            <tr valign="top">
                                <th scope="row">Patreon API Token</th>
                                <td>
                                    <input type="text" name="patreon_api_token" value="<?php echo esc_attr(get_option('patreon_api_token', '')); ?>" class="regular-text" />
                                    <p class="description">Enter your Patreon Creator API Token. This allows the plugin to check the user's membership level.</p>
                                </td>
                            </tr>

                        </table>
                        <?php submit_button(); ?>
                    </form>
                </div>
                <?php
            }
            /**
             * Fetch the user's Patreon pledge level.
             *
             * @param string $patreon_id The Patreon user ID.
             * @return string|false The pledge tier ID or false on failure.
             */
            private function get_patreon_pledge_level($patreon_id) {
                $api_token = get_option('patreon_api_token');
    
                if (!$api_token || !$patreon_id) {
                    return false;
                }
    
                $url = "https://www.patreon.com/api/oauth2/v2/members/$patreon_id?include=currently_entitled_tiers";
                $response = wp_remote_get($url, [
                    'headers' => [
                        'Authorization' => 'Bearer ' . $api_token,
                        'Content-Type' => 'application/json',
                    ],
                ]);
    
                if (is_wp_error($response)) {
                    return false;
                }
    
                $body = json_decode(wp_remote_retrieve_body($response), true);
    
                if (isset($body['data']['relationships']['currently_entitled_tiers']['data'][0]['id'])) {
                    return $body['data']['relationships']['currently_entitled_tiers']['data'][0]['id'];
                }
    
                return false;
            }
        }


        // Initialize the plugin
        new Patreon_User_Role_Changer_Plugin();
});
