(function() {
  const { __ } = wp.i18n;
  const { registerBlockType, createBlock } = wp.blocks;
  const { useState, useEffect } = wp.element;
  const { TextControl, ToggleControl, PanelBody, Button, SelectControl } = wp.components;
  const { InspectorControls, MediaUpload, MediaUploadCheck } = wp.blockEditor || wp.editor;

  registerBlockType('flw/secure-pdf', {
    title: __('Secure PDF', 'secure-pdf'),
    icon: 'media-document',
    category: 'embed',
    transforms: {
      from: [
        {
          type: 'shortcode',
          tag: 'secure_pdf',
          attributes: {
            url: { type: 'string', shortcode: 'url' },
            width: { type: 'string', shortcode: 'width' },
            height: { type: 'string', shortcode: 'height' },
            show_download: { type: 'string', shortcode: 'show_download' },
            show_print: { type: 'string', shortcode: 'show_print' },
            members_only: { type: 'string', shortcode: 'members_only' },
            mature_content: { type: 'string', shortcode: 'mature_content' },
            cover_image: { type: 'string', shortcode: 'cover_image' },
            open_button_text: { type: 'string', shortcode: 'open_button_text' },
            requires_purchase: { type: 'string', shortcode: 'requires_purchase' },
            price: { type: 'string', shortcode: 'price' },
          },
          transform: (attributes) => {
            const coerceBoolString = (val, fallback = 'false') => (val === 'true' || val === 'false') ? val : fallback;
            return createBlock('flw/secure-pdf', {
              url: attributes.url || '',
              width: attributes.width || '100%',
              height: attributes.height || '600px',
              show_download: coerceBoolString(attributes.show_download),
              show_print: coerceBoolString(attributes.show_print),
              members_only: coerceBoolString(attributes.members_only),
              mature_content: coerceBoolString(attributes.mature_content),
              cover_image: attributes.cover_image || '',
              open_button_text: attributes.open_button_text || '',
              requires_purchase: coerceBoolString(attributes.requires_purchase),
              price: attributes.price || '',
            });
          },
        },
      ],
    },
    attributes: {
      url: { type: 'string', default: '' },
      width: { type: 'string', default: '100%' },
      height: { type: 'string', default: '600px' },
      show_download: { type: 'string', default: 'false' },
      show_print: { type: 'string', default: 'false' },
      members_only: { type: 'string', default: 'false' },
      mature_content: { type: 'string', default: 'false' },
      cover_image: { type: 'string', default: '' },
      open_button_text: { type: 'string', default: '' },
      cover_fit: { type: 'string', default: 'cover' },
      requires_purchase: { type: 'string', default: 'false' },
      price: { type: 'string', default: '' },
    },
    edit: (props) => {
      const { attributes, setAttributes } = props;
      const [url, setUrl] = useState(attributes.url || '');

      useEffect(() => setAttributes({ url }), [url]);

      return (
        wp.element.createElement(
          'div',
          { className: 'secure-pdf-block-placeholder' },
          [
            wp.element.createElement('div', { key: 'label' }, __('Secure PDF Embedder', 'secure-pdf')),
            wp.element.createElement('div', { key: 'url-wrapper', style: { display: 'flex', gap: '8px', alignItems: 'flex-end' } }, [
              wp.element.createElement(TextControl, {
                key: 'url',
                label: __('PDF URL', 'secure-pdf'),
                placeholder: __('https://example.com/file.pdf', 'secure-pdf'),
                value: url,
                onChange: setUrl,
                style: { flex: 1 },
              }),
              wp.element.createElement(MediaUploadCheck, { key: 'media-check' },
                wp.element.createElement(MediaUpload, {
                  key: 'media-upload',
                  onSelect: (media) => {
                    if (!media) return;
                    // Prefer URL; if selecting an attachment with sizes, pick original
                    const selectedUrl = media.url || (media.source_url ? media.source_url : '');
                    if (selectedUrl) {
                      setUrl(selectedUrl);
                    }
                  },
                  multiple: false,
                  allowedTypes: ['application/pdf'],
                  render: ({ open }) =>
                    wp.element.createElement(Button, { isSecondary: true, onClick: open }, __('Media Library', 'secure-pdf')),
                })
              ),
            ]),
            wp.element.createElement(TextControl, {
              key: 'width',
              label: __('Width', 'secure-pdf'),
              value: attributes.width,
              onChange: (val) => setAttributes({ width: val }),
            }),
            wp.element.createElement(TextControl, {
              key: 'height',
              label: __('Height', 'secure-pdf'),
              help: __('Include units, e.g., 600px', 'secure-pdf'),
              value: attributes.height,
              onChange: (val) => setAttributes({ height: val }),
            }),
            // Quick toggles directly in the block panel
            wp.element.createElement(ToggleControl, {
              key: 'download-main',
              label: __('Show Download Button', 'secure-pdf'),
              checked: attributes.show_download === 'true',
              onChange: (val) => setAttributes({ show_download: val ? 'true' : 'false' }),
            }),
            wp.element.createElement(ToggleControl, {
              key: 'print-main',
              label: __('Show Print Button', 'secure-pdf'),
              checked: attributes.show_print === 'true',
              onChange: (val) => setAttributes({ show_print: val ? 'true' : 'false' }),
            }),
            wp.element.createElement(ToggleControl, {
              key: 'members-main',
              label: __('Members Only (UI)', 'secure-pdf'),
              checked: attributes.members_only === 'true',
              onChange: (val) => setAttributes({ members_only: val ? 'true' : 'false' }),
            }),
            wp.element.createElement(ToggleControl, {
              key: 'mature-main',
              label: __('Mature Content (Age Gate)', 'secure-pdf'),
              checked: attributes.mature_content === 'true',
              onChange: (val) => setAttributes({ mature_content: val ? 'true' : 'false' }),
            }),
            wp.element.createElement(ToggleControl, {
              key: 'purchase-required',
              label: __('Require Purchase (Paywall)', 'secure-pdf'),
              checked: attributes.requires_purchase === 'true',
              onChange: (val) => setAttributes({ requires_purchase: val ? 'true' : 'false' }),
            }),
            attributes.requires_purchase === 'true'
              ? wp.element.createElement(TextControl, {
                  key: 'price',
                  label: __('Price', 'secure-pdf'),
                  help: __('Shown like $10.00, sent to Stripe as cents', 'secure-pdf'),
                  placeholder: '$10.00',
                  value: attributes.price,
                  onChange: (val) => setAttributes({ price: val }),
                })
              : null,
            // Cover image preview, button, then Open button text (stacked)
            wp.element.createElement('div', { key: 'cover-controls', style: { display: 'flex', flexDirection: 'column', gap: '8px', alignItems: 'flex-start', marginTop: '8px' } }, [
              attributes.cover_image
                ? wp.element.createElement('img', {
                    key: 'cover-preview',
                    src: attributes.cover_image,
                    alt: __('Cover image preview', 'secure-pdf'),
                    style: { maxWidth: '100%', height: 'auto', maxHeight: '240px', border: '1px solid #ddd', borderRadius: '4px' }
                  })
                : null,
              wp.element.createElement(MediaUploadCheck, { key: 'cover-check' },
                wp.element.createElement(MediaUpload, {
                  key: 'cover-upload',
                  onSelect: (media) => {
                    if (!media) return;
                    const selectedUrl = media.url || (media.source_url ? media.source_url : '');
                    if (selectedUrl) {
                      setAttributes({ cover_image: selectedUrl });
                    }
                  },
                  multiple: false,
                  allowedTypes: ['image'],
                  render: ({ open }) =>
                    wp.element.createElement(Button, { isSecondary: true, onClick: open }, attributes.cover_image ? __('Change Cover Image', 'secure-pdf') : __('Set Cover Image', 'secure-pdf')),
                })
              ),
              wp.element.createElement(TextControl, {
                key: 'open-button-text',
                label: __('Open Button Text', 'secure-pdf'),
                help: __('Leave empty to use global setting', 'secure-pdf'),
                value: attributes.open_button_text,
                onChange: (val) => setAttributes({ open_button_text: val }),
              }),
              wp.element.createElement(SelectControl, {
                key: 'cover-fit',
                label: __('Cover image sizing', 'secure-pdf'),
                value: attributes.cover_fit || 'cover',
                options: [
                  { label: __('Cover (fill, crop if needed)', 'secure-pdf'), value: 'cover' },
                  { label: __('Fit width (100% width)', 'secure-pdf'), value: 'fit_width' },
                  { label: __('Fit height (100% height)', 'secure-pdf'), value: 'fit_height' },
                ],
                onChange: (val) => setAttributes({ cover_fit: val }),
              }),
            ]),
            // Editor preview removed: avoid loading viewer in admin editor
            wp.element.createElement('p', { key: 'placeholder' }, __('The PDF will render on the site. Editor preview is disabled.', 'secure-pdf')),
          ]
        )
      );
    },
    save: () => null, // Dynamic render via PHP
  });
})();


