console.log('DEBUGGING PDF');
console.log(pdfSettings);

// Mature content initialization and verification
function initMatureAndLoad() {
    const pdfContainer = document.getElementById('pdf-viewer');
    if (!pdfContainer) {
        console.error('PDF container not found');
        return;
    }
    // Guard against double-initialization (e.g., both inline and bundled DOMContentLoaded)
    if (pdfContainer.dataset.spdfInit === '1') {
        return;
    }
    pdfContainer.dataset.spdfInit = '1';
    const url = pdfContainer.dataset.url;
    const isMature = pdfContainer.dataset.mature === 'true';
    const coverImage = pdfContainer.dataset.coverImage || '';
    const openButtonText = pdfContainer.dataset.openButtonText || '';
    const requiresPurchase = pdfContainer.dataset.requiresPurchase === 'true';
    const priceCents = parseInt(pdfContainer.dataset.priceCents || '0', 10);
    const priceDisplay = pdfContainer.dataset.priceDisplay || '';
    const coverFit = pdfContainer.dataset.coverFit || 'cover';
    // Helper to check server if user already purchased
    const hasPurchased = async () => {
        try {
            const resp = await fetch(`${pdfSettings.ajaxurl}?action=secure_pdf_has_purchase&url=${encodeURIComponent(url)}`, { credentials: 'same-origin' });
            const data = await resp.json();
            return !!(data && data.success && data.data && data.data.purchased);
        } catch (e) { return false; }
    };

    // If purchase required, check first; if purchased, skip overlay and open immediately
    if (requiresPurchase) {
        hasPurchased().then((owned) => {
            if (owned) {
                // Reveal paywalled buttons and proceed
                document.querySelectorAll('.securepdf-paywall-button').forEach(btn => btn.classList.remove('securepdf-hidden'));
                if (isMature) { initMatureOnly(url, pdfContainer); } else { loadPDF(url); }
            } else {
                // Fall through to overlay flow (may include cover image)
                setupOverlay();
            }
        });
        return;
    }

    // If only cover image provided, set up overlay; otherwise just load
    if (coverImage) {
        setupOverlay();
        return;
    }

    // Default: no overlay; proceed
    if (isMature) { initMatureOnly(url, pdfContainer); } else { loadPDF(url); }

    function setupOverlay() {
        // If already wrapped by a previous init, reuse wrapper
        let wrapper = pdfContainer.closest('.cover-overlay-wrapper');
        if (!wrapper) {
            wrapper = document.createElement('div');
            wrapper.classList.add('cover-overlay-wrapper');
            const parent = pdfContainer.parentNode;
            parent.replaceChild(wrapper, pdfContainer);
            wrapper.appendChild(pdfContainer);
        }

        let overlay = wrapper.querySelector('.cover-overlay');
        if (!overlay) {
            overlay = document.createElement('div');
            overlay.classList.add('cover-overlay');
            let bgStyle = `background-image:url('${coverImage.replace(/'/g, "&#39;")}')`;
            if (coverFit === 'fit_width') bgStyle += ';background-size:contain;background-repeat:no-repeat;background-position:center top';
            else if (coverFit === 'fit_height') bgStyle += ';background-size:auto 100%;background-repeat:no-repeat;background-position:center center';
            else bgStyle += ';background-size:cover;background-position:center center';
            const bg = coverImage ? `<div class="cover-overlay-bg" style="${bgStyle}"></div>` : '';
            const label = (requiresPurchase && priceDisplay)
                ? `${openButtonText || 'Purchase to Open'} (${priceDisplay})`
                : (openButtonText || (requiresPurchase ? 'Purchase to Open' : 'Open Document'));
            const cta = requiresPurchase
                ? `<button class="cover-open-button" data-action="purchase">${label}</button>`
                : `<button class="cover-open-button">${label}</button>`;
            overlay.innerHTML = `${bg}${cta}`;
            wrapper.appendChild(overlay);
        }

        const openBtn = overlay.querySelector('.cover-open-button');
        if (openBtn) {
            openBtn.addEventListener('click', () => {
                if (requiresPurchase && openBtn.dataset.action === 'purchase') {
                    showStripeModal(url, pdfContainer, priceCents, () => {
                        overlay.remove();
                        const footer = wrapper.parentElement?.querySelector('.pdf-viewer-footer');
                        if (footer) footer.classList.remove('is-hidden');
                        // Unhide paywalled buttons after successful purchase
                        document.querySelectorAll('.securepdf-paywall-button').forEach(btn => btn.classList.remove('securepdf-hidden'));
                        if (isMature) { initMatureOnly(url, pdfContainer); } else { loadPDF(url); }
                    });
                    return;
                }
                overlay.remove();
                const footer = wrapper.parentElement?.querySelector('.pdf-viewer-footer');
                if (footer) footer.classList.remove('is-hidden');
                // If no purchase required, buttons should already be visible
                if (isMature) { initMatureOnly(url, pdfContainer); } else { loadPDF(url); }
            }, { once: true });
        }
        // Hide footer while overlay is present
        const footer = wrapper.parentElement?.querySelector('.pdf-viewer-footer');
        if (footer) footer.classList.add('is-hidden');
        return; // wait for user action
    }

    if (isMature) {
        // If user already verified for this URL in this browser, skip prompt
        try {
            const verifiedKey = `securepdf_verified_${btoa(url)}`;
            if (localStorage.getItem(verifiedKey) === 'true') {
                loadPDF(url);
                return;
            }
        } catch (e) {}

        // If already wrapped by a previous init, reuse that wrapper
        let wrapper = pdfContainer.closest('.mature-content-wrapper');
        if (!wrapper) {
            wrapper = document.createElement('div');
            wrapper.classList.add('mature-content-wrapper');
            const parent = pdfContainer.parentNode;
            parent.replaceChild(wrapper, pdfContainer);
            wrapper.appendChild(pdfContainer);
        }

        // Create and append overlay (ensure only one)
        let overlay = wrapper.querySelector('.mature-overlay');
        if (!overlay) {
            overlay = document.createElement('div');
            overlay.classList.add('mature-overlay');
            overlay.innerHTML = `
            <div class="mature-popup">
                <h2>Age Verification</h2>
                <p>Please enter your date of birth to view this content.</p>
                <input type="date" id="mature-dob" max="${new Date().toISOString().split('T')[0]}" />
                <div style="margin:8px 0; font-size: 0.9em; color: #555;">Format: YYYY-MM-DD or MM/DD/YYYY</div>
                <button id="verify-age">Verify</button>
                <p id="age-error" style="color: red; display: none;">You must be 18 or older to view this content.</p>
                <p id="age-invalid" style="color: red; display: none;">Please enter a valid date.</p>
            </div>
        `;
            wrapper.appendChild(overlay);
        }

        // Verification handler
        overlay.querySelector('#verify-age').addEventListener('click', function(e) {
            e.preventDefault();
            const dobInput = document.getElementById('mature-dob');
            const dobValueRaw = (dobInput.value || '').trim();
            const errorUnderage = document.getElementById('age-error');
            const errorInvalid = document.getElementById('age-invalid');
            errorUnderage.style.display = 'none';
            errorInvalid.style.display = 'none';

            if (!dobValueRaw) {
                errorInvalid.style.display = 'block';
                return;
            }

            let dob = null;
            if (/^\d{4}-\d{2}-\d{2}$/.test(dobValueRaw)) {
                const [y, m, d] = dobValueRaw.split('-').map(Number);
                dob = new Date(y, m - 1, d);
            } else if (/^\d{2}\/\d{2}\/\d{4}$/.test(dobValueRaw)) {
                const [m, d, y] = dobValueRaw.split('/').map(Number);
                dob = new Date(y, m - 1, d);
            } else {
                // Try native parse as last resort
                const parsed = new Date(dobValueRaw);
                if (!isNaN(parsed.getTime())) {
                    dob = parsed;
                }
            }

            if (!dob || isNaN(dob.getTime())) {
                errorInvalid.style.display = 'block';
                return;
            }
            const today = new Date();
            const age = today.getFullYear() - dob.getFullYear() - ((today.getMonth() < dob.getMonth() || (today.getMonth() === dob.getMonth() && today.getDate() < dob.getDate())) ? 1 : 0);
            if (age >= 18) {
                overlay.remove();
                // Remove class from any ancestor wrappers to clear blur/pointer rules
                let el = pdfContainer.parentElement;
                while (el) {
                    if (el.classList && el.classList.contains('mature-content-wrapper')) {
                        el.classList.remove('mature-content-wrapper');
                    }
                    el = el.parentElement;
                }
                try {
                    const verifiedKey = `securepdf_verified_${btoa(url)}`;
                    localStorage.setItem(verifiedKey, 'true');
                } catch (e) {}
                loadPDF(url);
            } else {
                errorUnderage.style.display = 'block';
            }
        });
    } else {
        loadPDF(url);
    }
}

// Mature-only gating used after cover overlay acceptance
function initMatureOnly(url, pdfContainer) {
    const isMature = pdfContainer.dataset.mature === 'true';
    if (!isMature) {
        loadPDF(url);
        return;
    }
    try {
        const verifiedKey = `securepdf_verified_${btoa(url)}`;
        if (localStorage.getItem(verifiedKey) === 'true') {
            loadPDF(url);
            return;
        }
    } catch (e) {}

    let wrapper = pdfContainer.closest('.mature-content-wrapper');
    if (!wrapper) {
        wrapper = document.createElement('div');
        wrapper.classList.add('mature-content-wrapper');
        const parent = pdfContainer.parentNode;
        parent.replaceChild(wrapper, pdfContainer);
        wrapper.appendChild(pdfContainer);
    }

    let overlay = wrapper.querySelector('.mature-overlay');
    if (!overlay) {
        overlay = document.createElement('div');
        overlay.classList.add('mature-overlay');
        overlay.innerHTML = `
        <div class="mature-popup">
            <h2>Age Verification</h2>
            <p>Please enter your date of birth to view this content.</p>
            <input type="date" id="mature-dob" max="${new Date().toISOString().split('T')[0]}" />
            <div style="margin:8px 0; font-size: 0.9em; color: #555;">Format: YYYY-MM-DD or MM/DD/YYYY</div>
            <button id="verify-age">Verify</button>
            <p id="age-error" style="color: red; display: none;">You must be 18 or older to view this content.</p>
            <p id="age-invalid" style="color: red; display: none;">Please enter a valid date.</p>
        </div>
    `;
        wrapper.appendChild(overlay);
    }

    overlay.querySelector('#verify-age').addEventListener('click', function(e) {
        e.preventDefault();
        const dobInput = document.getElementById('mature-dob');
        const dobValueRaw = (dobInput.value || '').trim();
        const errorUnderage = document.getElementById('age-error');
        const errorInvalid = document.getElementById('age-invalid');
        errorUnderage.style.display = 'none';
        errorInvalid.style.display = 'none';
        if (!dobValueRaw) {
            errorInvalid.style.display = 'block';
            return;
        }
        let dob = null;
        if (/^\d{4}-\d{2}-\d{2}$/.test(dobValueRaw)) {
            const [y, m, d] = dobValueRaw.split('-').map(Number);
            dob = new Date(y, m - 1, d);
        } else if (/^\d{2}\/\d{2}\/\d{4}$/.test(dobValueRaw)) {
            const [m, d, y] = dobValueRaw.split('/').map(Number);
            dob = new Date(y, m - 1, d);
        } else {
            const parsed = new Date(dobValueRaw);
            if (!isNaN(parsed.getTime())) {
                dob = parsed;
            }
        }
        if (!dob || isNaN(dob.getTime())) {
            errorInvalid.style.display = 'block';
            return;
        }
        const today = new Date();
        const age = today.getFullYear() - dob.getFullYear() - ((today.getMonth() < dob.getMonth() || (today.getMonth() === dob.getMonth() && today.getDate() < dob.getDate())) ? 1 : 0);
        if (age >= 18) {
            overlay.remove();
            let el = pdfContainer.parentElement;
            while (el) {
                if (el.classList && el.classList.contains('mature-content-wrapper')) {
                    el.classList.remove('mature-content-wrapper');
                }
                el = el.parentElement;
            }
            try {
                const verifiedKey = `securepdf_verified_${btoa(url)}`;
                localStorage.setItem(verifiedKey, 'true');
            } catch (e) {}
            loadPDF(url);
        } else {
            errorUnderage.style.display = 'block';
        }
    });
}

// Fetch progress from the server
function getProgress(url) {
    const requestUrl = `${pdfSettings.ajaxurl}?action=get_progress&url=${encodeURIComponent(url)}`;
    return fetch(requestUrl, {
        method: 'GET',
        credentials: 'same-origin',
    })
    .then(response => response.json())
    .then(data => {
        if (pdfSettings.debug) console.log('getProgress response:', data);
        if (data.success && data.data && data.data.pageNumber) {
            return data.data.pageNumber;
        }
        return 1;
    })
    .catch(error => {
        if (pdfSettings.debug) console.error('getProgress error:', error);
        return 1;
    });
}

// Save progress to the server
function saveProgress(url, pageNumber) {
    fetch(pdfSettings.ajaxurl, {
        method: 'POST',
        headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
        body: new URLSearchParams({
            action: 'save_progress',
            url: url,
            pageNumber: pageNumber,
        }),
        credentials: 'same-origin',
    })
    .then(response => response.json())
    .then(data => {
        if (pdfSettings.debug) console.log('saveProgress response:', data);
    })
    .catch(error => {
        if (pdfSettings.debug) console.error('saveProgress error:', error);
    });
}

// Render a specific page
function renderPage(pdfDoc, pageNumber, container, url) {
    // Skip if this page was already rendered into the container
    if (container.querySelector(`canvas[data-page-number="${pageNumber}"]`)) {
        return;
    }
    pdfDoc.getPage(pageNumber).then(page => {
        const canvas = document.createElement('canvas');
        const ctx = canvas.getContext('2d');
        const viewport = page.getViewport({ scale: 1.5 });
        canvas.width = viewport.width;
        canvas.height = viewport.height;
        canvas.dataset.pageNumber = pageNumber;
        container.appendChild(canvas);
        page.render({ canvasContext: ctx, viewport: viewport });

        const observer = new IntersectionObserver(([entry]) => {
            if (entry.isIntersecting) saveProgress(url, pageNumber);
        }, { threshold: 0.5 });
        observer.observe(canvas);
    });
}

// Load and render the PDF
function loadPDF(url) {
    const pdfContainer = document.getElementById('pdf-viewer');
    if (!pdfContainer) return;
    // Prevent duplicate full-document rendering
    if (pdfContainer.dataset.spdfLoaded === '1') {
        return;
    }
    pdfContainer.dataset.spdfLoaded = '1';
    getProgress(url).then(lastReadPage => {
        pdfjsLib.getDocument(url).promise.then(pdfDoc => {
            const totalPages = pdfDoc.numPages;
            for (let i = 1; i <= totalPages; i++) {
                renderPage(pdfDoc, i, pdfContainer, url);
            }
            // Do not auto-scroll the window. Instead, if available, scroll only inside
            // the scrollable viewer container to the last-read page.
            const containerEl = pdfContainer.parentElement; // .pdf-viewer-container
            if (containerEl && containerEl.classList.contains('pdf-viewer-container')) {
                const target = pdfContainer.querySelector(`[data-page-number="${lastReadPage}"]`);
                if (target) {
                    containerEl.scrollTo({ top: target.offsetTop, behavior: 'auto' });
                }
            }
        }).catch(err => console.error('Error loading PDF:', err));
    });
}

// DOM ready: initialize PDF viewer and shortcode generator
document.addEventListener('DOMContentLoaded', function() {
    const pdfViewer = document.getElementById('pdf-viewer');
    if (pdfViewer) {
        pdfViewer.addEventListener('contextmenu', e => e.preventDefault());
        initMatureAndLoad();
    }

    // Support for Gutenberg block editor live preview (when editing post)
    // If the block placeholder renders a container with data-url/data-mature, initialize when present
    const maybeInitBlock = () => {
        const blockViewer = document.querySelector('.wp-block[data-type="flw/secure-pdf"] #pdf-viewer');
        if (blockViewer && !blockViewer.dataset.initialized) {
            blockViewer.dataset.initialized = 'true';
            initMatureAndLoad();
        }
    };
    // Observe for inserted blocks during editor changes
    if (typeof wp !== 'undefined' && document.body.classList.contains('block-editor-page')) {
        const observer = new MutationObserver(() => maybeInitBlock());
        observer.observe(document.body, { childList: true, subtree: true });
        maybeInitBlock();
    }

    // Shortcode generator & copy functionality (unchanged)
    const generateButton = document.getElementById('generate-shortcode');
    const shortcodeOutput = document.getElementById('generated-shortcode');
    const copyButton = document.getElementById('copy-shortcode');
    const copySuccess = document.getElementById('copy-success');
    if (generateButton && shortcodeOutput) {
        generateButton.addEventListener('click', function() {
            const pdfUrl = document.getElementById('pdf-url').value.trim();
            const pdfWidth = document.getElementById('pdf-width').value.trim() || '100%';
            let pdfHeight = document.getElementById('pdf-height').value.trim() || '600px';
            if (/^\d+$/.test(pdfHeight)) {
                pdfHeight = `${pdfHeight}px`;
            }
            const showDownload = document.getElementById('show-download').checked ? 'true' : 'false';
            const membersOnly = document.getElementById('members-only').checked ? 'true' : 'false';
            const mature = document.getElementById('mature-content')?.checked ? 'true' : 'false';

            if (!pdfUrl) {
                shortcodeOutput.innerHTML = '<p style="color: red;">Please enter a valid PDF URL.</p>';
                copyButton.style.display = 'none';
                return;
            }

            let shortcode = `[secure_pdf url="${pdfUrl}" width="${pdfWidth}" height="${pdfHeight}" show_download="${showDownload}" members_only="${membersOnly}"`;
            if (mature === 'true') shortcode += ` mature_content="true"`;
            shortcode += `]`;

            shortcodeOutput.innerHTML = `<p><strong>Generated Shortcode:</strong></p><code id="shortcode-text">${shortcode}</code>`;
            copyButton.style.display = 'inline-block';
        });
    }
    if (copyButton) {
        copyButton.addEventListener('click', function() {
            const text = document.getElementById('shortcode-text');
            if (text) {
                const range = document.createRange();
                range.selectNode(text);
                window.getSelection().removeAllRanges();
                window.getSelection().addRange(range);
                document.execCommand('copy');
                window.getSelection().removeAllRanges();
                copySuccess.style.display = 'inline';
                setTimeout(() => copySuccess.style.display = 'none', 2000);
            }
        });
    }
});

// Simple Stripe modal stub (replace with real Stripe Elements integration)
function showStripeModal(url, pdfContainer, priceCents, onSuccess) {
    // Prevent multiple
    if (document.querySelector('.securepdf-modal-backdrop')) return;
    const backdrop = document.createElement('div');
    backdrop.className = 'securepdf-modal-backdrop';
    const modal = document.createElement('div');
    modal.className = 'securepdf-modal';
    const priceText = priceCents && priceCents > 0 ? ` ($${(priceCents/100).toFixed(2)})` : '';
    const mode = (pdfSettings && pdfSettings.stripeMode) ? String(pdfSettings.stripeMode) : '';
    const modeClass = mode === 'live' ? 'is-live' : (mode === 'test' ? 'is-test' : '');
    const showModeBadge = !!(mode && mode !== 'live');
    const isLoggedIn = !!(pdfSettings && (pdfSettings.loggedIn === 1 || pdfSettings.loggedIn === true || String(pdfSettings.loggedIn) === '1'));
    modal.innerHTML = `
        <h3>Purchase Required${priceText}</h3>
        <p>Enter payment details to access this document.</p>
        ${!isLoggedIn ? `
        <div id="securepdf-account">
            <div style="display:flex;gap:8px;flex-wrap:wrap;margin:8px 0 12px 0;">
                <input id="securepdf-first" type="text" placeholder="First name" style="flex:1;min-width:160px;padding:8px;border:1px solid #ddd;border-radius:4px;" />
                <input id="securepdf-last" type="text" placeholder="Last name" style="flex:1;min-width:160px;padding:8px;border:1px solid #ddd;border-radius:4px;" />
                <input id="securepdf-email" type="email" placeholder="Email" style="flex:1;min-width:200px;padding:8px;border:1px solid #ddd;border-radius:4px;" />
                <input id="securepdf-pass" type="password" placeholder="Password" style="flex:1;min-width:200px;padding:8px;border:1px solid #ddd;border-radius:4px;" />
            </div>
        </div>` : ''}
        <div id="securepdf-card-element" style="padding:12px 10px;border:1px solid #ddd;border-radius:4px;background:#fafafa"></div>
        <div id="securepdf-card-errors" role="alert" style="color:#d63638;min-height:1em;margin-top:8px;"></div>
        <div class="modal-actions">
            ${showModeBadge ? `<span class="securepdf-mode-indicator ${modeClass}">${mode.toUpperCase()} MODE</span>` : ''}
            <button id="securepdf-cancel" class="pdf-button">Cancel</button>
            <button id="securepdf-pay" class="pdf-button">Pay${priceText}</button>
        </div>
    `;
    backdrop.appendChild(modal);
    document.body.appendChild(backdrop);

    const close = () => backdrop.remove();
    document.getElementById('securepdf-cancel').addEventListener('click', close);
    const payBtn = document.getElementById('securepdf-pay');

    // Initialize Stripe Elements
    let stripe = null;
    let card = null;
    const errorEl = document.getElementById('securepdf-card-errors');
    if (window.Stripe && pdfSettings.stripePk) {
        try {
            stripe = window.Stripe(pdfSettings.stripePk);
            const elements = stripe.elements();
            card = elements.create('card', { hidePostalCode: true });
            card.mount('#securepdf-card-element');
            card.on('change', function(event){
                errorEl.textContent = event.error ? event.error.message : '';
            });
        } catch (e) {
            errorEl.textContent = 'Stripe failed to initialize.';
            payBtn.disabled = true;
        }
    } else {
        const modeHint = pdfSettings.stripeMode ? ` (mode: ${pdfSettings.stripeMode})` : '';
        errorEl.textContent = 'Stripe is not configured' + modeHint + '.';
        payBtn.disabled = true;
    }

    payBtn.addEventListener('click', async () => {
        try {
            if (!stripe || !card) throw new Error('Payment form unavailable');
            payBtn.disabled = true;
            const originalText = payBtn.textContent;
            payBtn.textContent = 'Processing…';
            // If not logged in, create account first
            if (!isLoggedIn && !(pdfSettings && (pdfSettings.loggedIn === 1 || String(pdfSettings.loggedIn) === '1'))) {
                const email = (document.getElementById('securepdf-email').value || '').trim();
                const pass = (document.getElementById('securepdf-pass').value || '').trim();
                const first = (document.getElementById('securepdf-first').value || '').trim();
                const last  = (document.getElementById('securepdf-last').value || '').trim();
                if (!email || !pass) throw new Error('Please enter email and password.');
                const reg = new FormData();
                reg.append('action', 'secure_pdf_register');
                reg.append('email', email);
                reg.append('password', pass);
                if (first) reg.append('first_name', first);
                if (last) reg.append('last_name', last);
                reg.append('_wpnonce', pdfSettings.nonce || '');
                const rresp = await fetch(pdfSettings.ajaxurl, { method: 'POST', body: reg, credentials: 'same-origin' });
                const rdata = await rresp.json();
                if (!rdata.success) {
                    // If account exists, prompt user to sign in and stop flow
                    if (rdata.data && (rdata.data.code === 'account_exists' || /exists/i.test(rdata.data.message||''))) {
                        errorEl.textContent = rdata.data.message || 'Account exists. Please sign in.';
                        payBtn.disabled = false; payBtn.textContent = `Pay${priceText}`; return; 
                    }
                    throw new Error(rdata.data && rdata.data.message ? rdata.data.message : 'Could not create account.');
                }
                // Consider logged-in for the remainder of the flow
                pdfSettings.loggedIn = 1;
            }
            // Create PaymentIntent via WordPress AJAX using server-side secret
            const form = new FormData();
            form.append('action', 'secure_pdf_create_payment_intent');
            form.append('amount', String(priceCents > 0 ? priceCents : 100));
            form.append('url', url);
            const resp = await fetch(pdfSettings.ajaxurl, { method: 'POST', body: form, credentials: 'same-origin' });
            const data = await resp.json();
            if (!data.success || !data.data || !data.data.client_secret) throw new Error('Failed to create PaymentIntent');

            const result = await stripe.confirmCardPayment(data.data.client_secret, {
                payment_method: { card }
            });
            if (result.error) throw result.error;
            if (!result.paymentIntent || result.paymentIntent.status !== 'succeeded') {
                throw new Error('Payment was not completed.');
            }

            // Mark purchase for logged-in users (no-op for guests)
            const mark = new FormData();
            mark.append('action', 'secure_pdf_mark_purchase');
            mark.append('url', url);
            mark.append('page', window.location.href);
            await fetch(pdfSettings.ajaxurl, { method: 'POST', body: mark, credentials: 'same-origin' });

            close();
            if (typeof onSuccess === 'function') onSuccess();
            // Ensure page reflects login/purchase state (header, buttons, etc.)
            try { window.location.reload(); } catch (e) {}
        } catch (e) {
            errorEl.textContent = e && e.message ? e.message : 'Payment failed. Please try again.';
            payBtn.disabled = false;
            payBtn.textContent = `Pay${priceText}`;
        }
    });
}
