<?php
/*
Plugin Name: Secure PDF Embedder
Description: Embed a secure PDF document in your WordPress site using a shortcode. Tracks user progress, restores the last-read page, and supports mature-content age verification.
Version: 2.9.9
Author: FrostLine Works
Author URI: https://frostlineworks.com
Requires Plugins: flwsecureupdates
*/

if (!defined('ABSPATH')) {
    exit;
}

    /** Enqueue scripts & styles for both admin and frontend */
    function secure_pdf_enqueue_scripts($hook = '') {
        if (!function_exists('get_file_data')) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }
        $plugin_data    = get_file_data(__FILE__, ['Version' => 'Version']);
        $script_version = $plugin_data['Version'];
        wp_enqueue_style(
            'secure-pdf-style',
            plugins_url('secure-pdf.css', __FILE__),
            [],
            filemtime(plugin_dir_path(__FILE__) . 'secure-pdf.css')
        );
        $script_handle = 'secure-pdf-script';
        $script_url    = plugins_url('secure-pdf.js', __FILE__);
        if (is_admin() && $hook === 'flw-plugins_page_securepdf') {
            wp_enqueue_script($script_handle, $script_url, ['jquery'], $script_version, true);
            wp_localize_script($script_handle, 'pdfSettings', [
                'debug' => (defined('WP_DEBUG') && WP_DEBUG),
            ]);
            // Color picker assets for settings UI
            wp_enqueue_style('wp-color-picker');
            wp_enqueue_script('wp-color-picker');
            $picker_init = "jQuery(function($){\n".
                "  $('.secure-pdf-color-picker').each(function(){\n".
                "    var input = $(this);\n".
                "    var def = input.data('default-color') || input.attr('placeholder') || '#ffffff';\n".
                "    input.wpColorPicker({ defaultColor: def });\n".
                "  });\n".
                "  function refreshState(wrapper){\n".
                "    var disabled = wrapper.find('.secure-pdf-transparent').is(':checked');\n".
                "    wrapper.find('.secure-pdf-color-picker').prop('disabled', disabled);\n".
                "    wrapper.find('.wp-color-picker').prop('disabled', disabled);\n".
                "    wrapper.find('.wp-picker-clear, .wp-picker-default, .wp-color-result').prop('disabled', disabled);\n".
                "    wrapper.find('.secure-pdf-opacity').prop('disabled', disabled);\n".
                "  }\n".
                "  $('.secure-pdf-opacity').on('input change', function(){\n".
                "    $(this).closest('.secure-pdf-color-control').find('.secure-pdf-opacity-value').text($(this).val() + '%');\n".
                "  }).trigger('change');\n".
                "  $('.secure-pdf-transparent').on('change', function(){\n".
                "    refreshState($(this).closest('.secure-pdf-color-control'));\n".
                "  }).each(function(){\n".
                "    refreshState($(this).closest('.secure-pdf-color-control'));\n".
                "  });\n".
                "});";
            wp_add_inline_script($script_handle, $picker_init, 'after');
        } else {
            // Frontend: only enqueue when the shortcode is present on the current singular post/page
            if (!is_singular()) {
                return;
            }
            global $post;
            $has_shortcode = ($post && isset($post->post_content) && has_shortcode($post->post_content, 'secure_pdf'));
            $has_block     = function_exists('has_block') && has_block('flw/secure-pdf', $post);
            if (!$post || (!$has_shortcode && !$has_block)) {
                return;
            }

            wp_enqueue_script('pdfjs', 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/2.16.105/pdf.min.js', [], null, true);
            // Ensure the worker is set for pdf.js
            wp_add_inline_script(
                'pdfjs',
                "if (window.pdfjsLib) { pdfjsLib.GlobalWorkerOptions.workerSrc = 'https://cdnjs.cloudflare.com/ajax/libs/pdf.js/2.16.105/pdf.worker.min.js'; }",
                'after'
            );

            // Stripe.js for paywall flows
            wp_enqueue_script('stripe-js', 'https://js.stripe.com/v3', [], null, true);
            wp_enqueue_script($script_handle, $script_url, ['pdfjs', 'jquery', 'stripe-js'], $script_version, true);
            $pk = secure_pdf_get_stripe_publishable_key();
            $mode = secure_pdf_get_stripe_mode();
            if (!empty($pk)) {
                $prefix = substr($pk, 0, 7);
                if ($prefix === 'pk_live') { $mode = 'live'; }
                elseif ($prefix === 'pk_test') { $mode = 'test'; }
            }
            wp_localize_script($script_handle, 'pdfSettings', [
                'debug'   => (defined('WP_DEBUG') && WP_DEBUG),
                'ajaxurl' => admin_url('admin-ajax.php'),
                'stripePk' => $pk,
                'stripeMode' => $mode,
                'loggedIn' => is_user_logged_in() ? 1 : 0,
                'nonce' => wp_create_nonce('secure_pdf'),
            ]);
        }
    }
    add_action('wp_enqueue_scripts', 'secure_pdf_enqueue_scripts');
    add_action('admin_enqueue_scripts', 'secure_pdf_enqueue_scripts');

    /** Register Gutenberg block that maps to the shortcode output (dynamic render) */
    function secure_pdf_register_block() {
        // Editor script for block controls
        wp_register_script(
            'secure-pdf-block',
            plugins_url('block.js', __FILE__),
            ['wp-blocks', 'wp-element', 'wp-i18n', 'wp-block-editor', 'wp-components', 'wp-editor'],
            filemtime(plugin_dir_path(__FILE__) . 'block.js'),
            true
        );

        // Optional: minimal editor styles for the placeholder
        wp_register_style(
            'secure-pdf-block-editor',
            plugins_url('block.css', __FILE__),
            [],
            filemtime(plugin_dir_path(__FILE__) . 'block.css')
        );

        register_block_type('flw/secure-pdf', [
            'editor_script' => 'secure-pdf-block',
            'editor_style'  => 'secure-pdf-block-editor',
            // Frontend style is already handled by secure-pdf-style
            'render_callback' => function($attributes) {
                // Reuse shortcode rendering for consistency
                return secure_pdf_embed_shortcode($attributes);
            },
            'attributes' => [
                'url' => [ 'type' => 'string', 'default' => '' ],
                'width' => [ 'type' => 'string', 'default' => '100%' ],
                'height' => [ 'type' => 'string', 'default' => '600px' ],
                'show_download' => [ 'type' => 'string', 'default' => 'false' ],
                'show_print' => [ 'type' => 'string', 'default' => 'false' ],
                'members_only' => [ 'type' => 'string', 'default' => 'false' ],
                'mature_content' => [ 'type' => 'string', 'default' => 'false' ],
                'cover_image' => [ 'type' => 'string', 'default' => '' ],
                'open_button_text' => [ 'type' => 'string', 'default' => '' ],
                'cover_fit' => [ 'type' => 'string', 'default' => 'cover' ],
                'requires_purchase' => [ 'type' => 'string', 'default' => 'false' ],
                'price' => [ 'type' => 'string', 'default' => '' ],
            ],
        ]);
    }
    /** Helper: parse a user-facing price string (e.g., "$10.00", "10", "10.5") to cents */
    function secure_pdf_price_to_cents($price_str) {
        $s = trim((string)$price_str);
        if ($s === '') return 0;
        // Remove currency symbols and commas
        $s = preg_replace('/[^0-9.]/', '', $s);
        if ($s === '' || $s === '.') return 0;
        $float = floatval($s);
        if (!is_finite($float) || $float < 0) return 0;
        return (int) round($float * 100);
    }

    /** Helper: resolve Stripe mode based on external plugin options */
    function secure_pdf_get_stripe_mode() {
        // If local/site keys are present, prefer local toggle for FRONTEND (publishable key)
        $has_local = (get_option('flw_stripe_mode_live', null) !== null)
            || (get_option('flw_stripe_test_public_key', null) !== null)
            || (get_option('flw_stripe_live_public_key', null) !== null)
            || (get_option('flw_stripe_public_key', null) !== null);

        if ($has_local) {
            $local_live = (bool) get_option('flw_stripe_mode_live');
            $mode = $local_live ? 'live' : 'test';
            return apply_filters('secure_pdf/stripe_mode', $mode);
        }

        // Otherwise, fall back to remote/broker mode
        if (function_exists('flw_su_stripe_mode')) {
            $m = (string) flw_su_stripe_mode();
            $mode = ($m === 'live') ? 'live' : 'test';
            return apply_filters('secure_pdf/stripe_mode', $mode);
        }
        $opt_remote = get_option('flw_su_stripe_mode', 'test');
        if ($opt_remote === 'live' || $opt_remote === 'test') {
            return apply_filters('secure_pdf/stripe_mode', $opt_remote);
        }
        // Default
        return apply_filters('secure_pdf/stripe_mode', 'test');
    }

    /** Helper: retrieve Stripe publishable key from the external plugin/options */
    function secure_pdf_get_stripe_publishable_key() {
        // Common helper function names across environments
        $funcs = [
            'flw_su_get_stripe_publishable',
            'flw_su_get_publishable_key',
            'flw_su_publishable_key',
            'flw_get_stripe_publishable_key',
            'flw_stripe_publishable_key',
        ];
        foreach ($funcs as $fn) {
            if (function_exists($fn)) {
                $pk = (string) call_user_func($fn);
                if (!empty($pk)) return apply_filters('secure_pdf/stripe_publishable_key', $pk, secure_pdf_get_stripe_mode());
            }
        }
        $mode = secure_pdf_get_stripe_mode();
        $candidates = [];
        if ($mode === 'live') {
            $candidates = [
                'flw_su_stripe_live_pk',
                'flw_su_live_pk',
                'flw_su_stripe_publishable_live',
                'flw_su_publishable_live',
                'flw_stripe_live_pk',
                'flw_stripe_publishable_live',
                'flw_su_stripe_pk_live',
                'flw_stripe_pk_live',
                'flw_stripe_live_public_key',
            ];
        } else {
            $candidates = [
                'flw_su_stripe_test_pk',
                'flw_su_test_pk',
                'flw_su_stripe_publishable_test',
                'flw_su_publishable_test',
                'flw_stripe_test_pk',
                'flw_stripe_publishable_test',
                'flw_su_stripe_pk_test',
                'flw_stripe_pk_test',
                'flw_stripe_test_public_key',
            ];
        }
        foreach ($candidates as $opt) {
            $val = (string) get_option($opt, '');
            if (!empty($val)) return apply_filters('secure_pdf/stripe_publishable_key', $val, $mode);
        }
        // Fallback setting provided by this plugin, if admin entered
        $fallbacks = [
            'secure_pdf_stripe_publishable_key', // our plugin fallback
            'flw_stripe_public_key',            // legacy
        ];
        foreach ($fallbacks as $opt) {
            $val = (string) get_option($opt, '');
            if (!empty($val)) return apply_filters('secure_pdf/stripe_publishable_key', $val, $mode);
        }
        return apply_filters('secure_pdf/stripe_publishable_key', '', $mode);
    }

    /** Helper: retrieve Stripe secret key for server-side calls */
    function secure_pdf_get_stripe_secret_key() {
        // Prefer helper if available
        if (function_exists('flw_su_get_stripe_secret')) {
            $sec = (string) flw_su_get_stripe_secret();
            if (!empty($sec)) return $sec;
        }
        // Remote/broker options
        $remote_mode = get_option('flw_su_stripe_mode', null);
        if ($remote_mode !== null) {
            $is_live = ($remote_mode === 'live');
            $sec = $is_live ? (string) get_option('flw_su_stripe_api_key_live', '') : (string) get_option('flw_su_stripe_api_key_test', '');
            if (!empty($sec)) return $sec;
        }
        // Local/site options
        $is_live_local = (bool) get_option('flw_stripe_mode_live');
        $sec = $is_live_local ? (string) get_option('flw_stripe_live_secret_key', '') : (string) get_option('flw_stripe_test_secret_key', '');
        if (!empty($sec)) return $sec;
        // Legacy
        $legacy = (string) get_option('flw_stripe_secret_key', '');
        if (!empty($legacy)) return $legacy;
        // Nothing found
        return '';
    }

    add_action('init', 'secure_pdf_register_block');

    // Ensure media modal assets are available in the block editor
    add_action('admin_enqueue_scripts', function ($hook) {
        if (!function_exists('get_current_screen')) {
            return;
        }
        $screen = get_current_screen();
        if ($screen && method_exists($screen, 'is_block_editor') && $screen->is_block_editor()) {
            wp_enqueue_media();
        }
    });

    /** Add settings link on plugins page */
    add_filter('plugin_action_links_' . plugin_basename(__FILE__), function ($links) {
        $pluginSlug   = basename(dirname(__FILE__));
        $settings_url = admin_url('admin.php?page=' . $pluginSlug);
        array_unshift($links, '<a href="' . esc_url($settings_url) . '">Settings</a>');
        return $links;
    });

    /** Helper: resolve color option into final CSS color (hex/rgba/transparent) */
    function secure_pdf_get_color_value($base_option, $default_hex) {
        $transparent = get_option($base_option . '_transparent', '0') === '1';
        if ($transparent) {
            return 'transparent';
        }
        $hex = trim((string) get_option($base_option, $default_hex));
        if ($hex === '') {
            $hex = $default_hex;
        }
        if ($hex[0] !== '#') {
            return $hex; // allow named colors or rgba() saved directly
        }
        $hex = ltrim($hex, '#');
        if (strlen($hex) === 3) {
            $hex = $hex[0].$hex[0].$hex[1].$hex[1].$hex[2].$hex[2];
        }
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        $opacity = floatval(get_option($base_option . '_opacity', '100'));
        $alpha = max(0.0, min(1.0, $opacity / 100.0));
        if ($alpha >= 1.0) {
            return '#' . $hex;
        }
        return 'rgba(' . $r . ',' . $g . ',' . $b . ',' . $alpha . ')';
    }

    /** Shortcode to embed PDF with optional mature-content flag */
    function secure_pdf_embed_shortcode($atts) {
        $atts = shortcode_atts([
            'url'              => '',
            'width'            => '100%',
            'height'           => '600px',
            'show_download'    => 'false',
            'show_print'       => 'false',
            'members_only'     => 'false',
            'mature_content'   => 'false',
            'cover_image'      => '',
            'open_button_text' => '',
            'cover_fit'        => 'cover',
            'requires_purchase'=> 'false',
            'price'            => '',
        ], $atts);

        if (empty($atts['url'])) {
            return '<p>' . __('Please provide a valid PDF URL.', 'secure-pdf') . '</p>';
        }

        // Retrieve styling options (with transparency/opacity support)
        $button_color              = secure_pdf_get_color_value('secure_pdf_button_color', '#0073aa');
        $button_hover_color        = secure_pdf_get_color_value('secure_pdf_button_hover_color', '#005177');
        $button_font_color         = secure_pdf_get_color_value('secure_pdf_button_font_color', '#ffffff');
        $footer_bg_color           = secure_pdf_get_color_value('secure_pdf_footer_background_color', '#333');
        $footer_border_color       = secure_pdf_get_color_value('secure_pdf_footer_top_border_color', '#000');
        $footer_border_thickness   = get_option('secure_pdf_footer_top_border_thickness', '4');
        $footer_border_style       = get_option('secure_pdf_footer_top_border_style', 'solid');
        $viewer_border_thickness   = get_option('secure_pdf_viewer_border_thickness', '1');
        $viewer_border_style       = get_option('secure_pdf_viewer_border_style', 'solid');
        $viewer_outline_color      = secure_pdf_get_color_value('secure_pdf_viewer_outline_color', '#ccc');
        $download_text             = get_option('secure_pdf_download_button_text', 'Download');
        $join_url                  = get_option('secure_pdf_join_now_url', '#');
        $join_text                 = get_option('secure_pdf_join_now_button_text', 'Join Now');

        $file_name = basename(parse_url($atts['url'], PHP_URL_PATH));
        $mature_flag = $atts['mature_content'] === 'true' ? 'true' : 'false';

        // Wrapper-level CSS variables to control styling globally
        $wrapper_vars = sprintf(
            '--button-bg:%1$s; --button-hover-bg:%2$s; --button-font:%3$s; --footer-bg:%4$s; --footer-border-thickness:%5$spx; --footer-border-style:%6$s; --footer-border-color:%7$s; --viewer-border-thickness:%8$spx; --viewer-border-style:%9$s; --viewer-outline-color:%10$s; --join-button-bg:%11$s; --join-button-font:%12$s; --open-button-bg:%13$s; --open-button-hover-bg:%14$s; --open-button-font:%15$s;',
            esc_attr($button_color),
            esc_attr($button_hover_color),
            esc_attr($button_font_color),
            esc_attr($footer_bg_color),
            intval($footer_border_thickness),
            esc_attr($footer_border_style),
            esc_attr($footer_border_color),
            intval($viewer_border_thickness),
            esc_attr($viewer_border_style),
            esc_attr($viewer_outline_color),
            esc_attr(secure_pdf_get_color_value('secure_pdf_join_button_color', '#d63638')),
            esc_attr(secure_pdf_get_color_value('secure_pdf_join_button_font_color', '#ffffff')),
            esc_attr(secure_pdf_get_color_value('secure_pdf_open_button_color', '#0073aa')),
            esc_attr(secure_pdf_get_color_value('secure_pdf_open_button_hover_color', '#005177')),
            esc_attr(secure_pdf_get_color_value('secure_pdf_open_button_font_color', '#ffffff'))
        );

        $output  = '<div class="pdf-viewer-wrapper" style="' . $wrapper_vars . '">';

        // Determine view permission when members_only is enabled
        $can_view = true;
        $is_members_only = ($atts['members_only'] === 'true');
        if ($is_members_only) {
            $user = wp_get_current_user();
            $is_member = ($user && (in_array('patreon_member', (array) $user->roles, true) || in_array('administrator', (array) $user->roles, true)));
            if (!$is_member) {
                $can_view = false;
            }
        }

        $output .= '<div class="pdf-viewer-container" style="width:' . esc_attr($atts['width']) . '; height:' . esc_attr($atts['height']) . ';">';
        if ($can_view) {
            $cover = trim((string)$atts['cover_image']);
            $btn_text = trim((string)$atts['open_button_text']);
            if ($btn_text === '') {
                $btn_text = get_option('secure_pdf_open_button_text', __('Open Document', 'secure-pdf'));
            }

            $has_cover = ($cover !== '');
            $data_attrs = ' data-url="' . esc_url($atts['url']) . '" data-mature="' . esc_attr($mature_flag) . '"';
            if ($has_cover) {
                $fit = in_array($atts['cover_fit'], ['cover','fit_width','fit_height'], true) ? $atts['cover_fit'] : 'cover';
                $data_attrs .= ' data-cover-image="' . esc_url($cover) . '" data-open-button-text="' . esc_attr($btn_text) . '" data-cover-fit="' . esc_attr($fit) . '"';
            }
            if ($atts['requires_purchase'] === 'true') {
                $data_attrs .= ' data-requires-purchase="true"';
                $price_cents = secure_pdf_price_to_cents($atts['price']);
                if ($price_cents > 0) {
                    $data_attrs .= ' data-price-cents="' . intval($price_cents) . '" data-price-display="' . esc_attr($atts['price']) . '"';
                }
            }
            $output .= '<div id="pdf-viewer" class="pdf-viewer"' . $data_attrs . '></div>';
        } else {
            $output .= '<div class="pdf-viewer" style="display:flex;align-items:center;justify-content:center;height:100%;text-align:center;">';
            $output .= '<div>';
            $output .= '<p>' . esc_html__('This content is for members only.', 'secure-pdf') . '</p>';
            $output .= '</div>';
            $output .= '</div>';
        }
        $output .= '</div>';

        // Footer
        $output .= '<div class="pdf-viewer-footer" style="width:' . esc_attr($atts['width']) . ';">';
        $output .= '<span class="pdf-filename">' . esc_html($file_name) . '</span>';

        if ($is_members_only) {
            if ($can_view) {
                $paywall_class = ($atts['requires_purchase'] === 'true') ? ' securepdf-paywall-button securepdf-hidden' : '';
                if ($atts['show_download'] === 'true') {
                    $download_href = ($atts['requires_purchase'] === 'true')
                        ? add_query_arg([
                            'action' => 'secure_pdf_download',
                            'url' => rawurlencode($atts['url']),
                            'redirect' => rawurlencode((is_ssl() ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']),
                          ], admin_url('admin-ajax.php'))
                        : $atts['url'];
                    $output .= '<button class="pdf-button pdf-download-button' . $paywall_class . '" onclick="window.open(\'' . esc_url($download_href) . '\', \'_blank\')">' . esc_html($download_text) . '</button>';
                }
                if ($atts['show_print'] === 'true') {
                    $output .= '<button class="pdf-button pdf-print-button' . $paywall_class . '" onclick="window.open(\'' . esc_url($atts['url']) . '\', \'_blank\')">' . esc_html__('Print', 'secure-pdf') . '</button>';
                }
            } else {
                $output .= '<button class="pdf-button pdf-join-button" onclick="window.open(\'' . esc_url($join_url) . '\', \'_blank\')">' . esc_html($join_text) . '</button>';
            }
        } else {
            $paywall_class = ($atts['requires_purchase'] === 'true') ? ' securepdf-paywall-button securepdf-hidden' : '';
            if ($atts['show_download'] === 'true') {
                $download_href = ($atts['requires_purchase'] === 'true')
                    ? add_query_arg([
                        'action' => 'secure_pdf_download',
                        'url' => rawurlencode($atts['url']),
                        'redirect' => rawurlencode((is_ssl() ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']),
                      ], admin_url('admin-ajax.php'))
                    : $atts['url'];
                $output .= '<button class="pdf-button pdf-download-button' . $paywall_class . '" onclick="window.open(\'' . esc_url($download_href) . '\', \'_blank\')">' . esc_html($download_text) . '</button>';
            }
            if ($atts['show_print'] === 'true') {
                $output .= '<button class="pdf-button pdf-print-button' . $paywall_class . '" onclick="window.open(\'' . esc_url($atts['url']) . '\', \'_blank\')">' . esc_html__('Print', 'secure-pdf') . '</button>';
            }
        }
        $output .= '</div>'; // .pdf-viewer-footer
        $output .= '</div>'; // .pdf-viewer-wrapper

        // Trigger JS load (JS handles mature logic) only if viewer is visible
        if ($can_view) {
            $output .= '<script>document.addEventListener("DOMContentLoaded", function(){initMatureAndLoad();});</script>';
        }
        return $output;
    }
    add_shortcode('secure_pdf', 'secure_pdf_embed_shortcode');

    /** AJAX handlers: save and retrieve progress */
    function secure_pdf_save_progress() {
        if (!isset($_POST['url'], $_POST['pageNumber'])) {
            wp_send_json_error(['message' => 'Invalid parameters.'], 400);
        }
        $url  = sanitize_text_field($_POST['url']);
        $page = intval($_POST['pageNumber']);
        $user = get_current_user_id();
        if (!$user) {
            wp_send_json_error(['message' => 'User not logged in.'], 403);
        }
        update_user_meta($user, 'pdf_progress_' . md5($url), $page);
        wp_send_json_success(['message' => "Progress saved for page {$page}."]);
    }
    add_action('wp_ajax_save_progress', 'secure_pdf_save_progress');

    function secure_pdf_get_progress() {
        if (!isset($_GET['url'])) {
            wp_send_json_error(['message' => 'Missing URL parameter.'], 400);
        }
        $url  = sanitize_text_field($_GET['url']);
        $user = get_current_user_id();
        if (!$user) {
            wp_send_json_error(['message' => 'User not logged in.'], 403);
        }
        $page = (int) get_user_meta($user, 'pdf_progress_' . md5($url), true) ?: 1;
        wp_send_json_success(['pageNumber' => $page]);
    }
    add_action('wp_ajax_get_progress', 'secure_pdf_get_progress');
    add_action('wp_ajax_nopriv_get_progress', 'secure_pdf_get_progress');

    // Paywall AJAX endpoints (minimal placeholders). Use secrets from external plugin if available
    add_action('wp_ajax_secure_pdf_mark_purchase', function() {
        if (!isset($_POST['url'])) {
            wp_send_json_error(['message' => 'Missing URL parameter.'], 400);
        }
        $url = esc_url_raw($_POST['url']);
        $page_url = isset($_POST['page']) ? esc_url_raw($_POST['page']) : '';
        $user_id = get_current_user_id();
        if ($user_id) {
            $key = 'securepdf_purchased_' . md5($url);
            $already = get_user_meta($user_id, $key, true);
            if (!$already) {
                update_user_meta($user_id, $key, 1);

                // Send confirmation email
                $user = get_userdata($user_id);
                if ($user && $user->user_email) {
                    $site_name = wp_specialchars_decode(get_bloginfo('name'), ENT_QUOTES);
                    $file_name = basename(parse_url($url, PHP_URL_PATH));
                    $to = $user->user_email;
                    $subject = sprintf(__('Your purchase on %s', 'secure-pdf'), $site_name);
                    $message = '<p>' . sprintf(__('Hi %s,', 'secure-pdf'), esc_html($user->display_name ?: $user->user_login)) . '</p>';
                    $message .= '<p>' . __('Thank you for your purchase. Your document is ready.', 'secure-pdf') . '</p>';
                    if ($page_url) {
                        $message .= '<p><strong>' . __('View on site:', 'secure-pdf') . '</strong> <a href="' . esc_url($page_url) . '">' . esc_html($page_url) . '</a></p>';
                    }
                    $download_gate = add_query_arg([
                        'action' => 'secure_pdf_download',
                        'url' => rawurlencode($url),
                        'redirect' => rawurlencode($page_url),
                    ], admin_url('admin-ajax.php'));
                    $message .= '<p><strong>' . __('Download link:', 'secure-pdf') . '</strong> <a href="' . esc_url($download_gate) . '">' . esc_html($file_name) . '</a></p>';
                    $message .= '<p>' . sprintf(__('If you have any issues, reply to this email or visit %s.', 'secure-pdf'), esc_html(home_url())) . '</p>';
                    $headers = ['Content-Type: text/html; charset=UTF-8'];
                    $email = apply_filters('secure_pdf_purchase_email', [
                        'to' => $to,
                        'subject' => $subject,
                        'message' => $message,
                        'headers' => $headers,
                    ], $user, $url, $page_url);
                    if (!empty($email['to']) && !empty($email['subject'])) {
                        wp_mail($email['to'], $email['subject'], $email['message'], $email['headers']);
                    }
                }
            }
        }
        wp_send_json_success(['marked' => true]);
    });
    add_action('wp_ajax_nopriv_secure_pdf_mark_purchase', function() {
        wp_send_json_success(['marked' => true]);
    });

    add_action('wp_ajax_secure_pdf_has_purchase', function() {
        if (!isset($_GET['url'])) {
            wp_send_json_error(['message' => 'Missing URL parameter.'], 400);
        }
        $url = sanitize_text_field($_GET['url']);
        $user = get_current_user_id();
        $purchased = false;
        if ($user) {
            $purchased = (bool) get_user_meta($user, 'securepdf_purchased_' . md5($url), true);
        }
        wp_send_json_success(['purchased' => $purchased]);
    });
    add_action('wp_ajax_nopriv_secure_pdf_has_purchase', function() {
        wp_send_json_success(['purchased' => false]);
    });

    // Download gate: requires login and prior purchase before redirecting to the PDF URL
    function secure_pdf_download_gate() {
        $pdf_url = isset($_GET['url']) ? esc_url_raw(rawurldecode((string) $_GET['url'])) : '';
        $redirect_page = isset($_GET['redirect']) ? esc_url_raw(rawurldecode((string) $_GET['redirect'])) : '';
        if (!$pdf_url) {
            wp_die(__('Missing download URL.', 'secure-pdf'), 400);
        }
        $current = (is_ssl() ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI'];
        if (!is_user_logged_in()) {
            wp_safe_redirect(wp_login_url($current));
            exit;
        }
        $user_id = get_current_user_id();
        $owned = (bool) get_user_meta($user_id, 'securepdf_purchased_' . md5($pdf_url), true);
        if (!$owned) {
            if ($redirect_page) {
                wp_safe_redirect($redirect_page);
            } else {
                wp_die(__('Purchase required to download this file.', 'secure-pdf'), 403);
            }
            exit;
        }
        // Create a short-lived token that maps to the real URL, then redirect to stream endpoint
        $token = wp_generate_password(24, false, false);
        set_transient('securepdf_dl_' . $token, $pdf_url, 5 * MINUTE_IN_SECONDS);
        $stream_url = add_query_arg(['action' => 'secure_pdf_stream', 'token' => $token], admin_url('admin-ajax.php'));
        wp_safe_redirect($stream_url);
        exit;
    }
    add_action('wp_ajax_secure_pdf_download', 'secure_pdf_download_gate');
    add_action('wp_ajax_nopriv_secure_pdf_download', 'secure_pdf_download_gate');

    // Stream the PDF without exposing its true URL. Requires login and valid token.
    function secure_pdf_stream_file() {
        if (!isset($_GET['token'])) {
            wp_die(__('Missing token.', 'secure-pdf'), 400);
        }
        if (!is_user_logged_in()) {
            wp_die(__('You must be logged in to download this file.', 'secure-pdf'), 403);
        }
        $token = sanitize_text_field(wp_unslash($_GET['token']));
        $transient_key = 'securepdf_dl_' . $token;
        $pdf_url = get_transient($transient_key);
        if (!$pdf_url) {
            wp_die(__('This link has expired. Please request a new download link from your email.', 'secure-pdf'), 410);
        }
        // One-time use
        delete_transient($transient_key);

        // Verify ownership again
        $user_id = get_current_user_id();
        if (!$user_id || !get_user_meta($user_id, 'securepdf_purchased_' . md5($pdf_url), true)) {
            wp_die(__('Purchase required to download this file.', 'secure-pdf'), 403);
        }

        // Determine filename
        $file_name = basename(parse_url($pdf_url, PHP_URL_PATH));
        if ($file_name === '' || strpos($file_name, '.') === false) {
            $file_name = 'document.pdf';
        }

        // Attempt local streaming when URL is within uploads dir; otherwise proxy fetch
        $uploads = wp_upload_dir();
        $basedir = $uploads['basedir'];
        $baseurl = $uploads['baseurl'];
        $served = false;
        if ($baseurl && strpos($pdf_url, $baseurl) === 0) {
            $relative = ltrim(substr($pdf_url, strlen($baseurl)), '/');
            $path = wp_normalize_path(trailingslashit($basedir) . $relative);
            if (file_exists($path) && is_readable($path)) {
                if (!headers_sent()) {
                    header('Content-Type: application/pdf');
                    header('Content-Disposition: attachment; filename="' . rawurlencode($file_name) . '"');
                    header('X-Accel-Buffering: no');
                }
                readfile($path);
                $served = true;
            }
        }
        if (!$served) {
            // Remote or fallback
            $resp = wp_remote_get($pdf_url, [ 'timeout' => 30 ]);
            if (is_wp_error($resp)) {
                wp_die(__('Unable to retrieve file.', 'secure-pdf'), 502);
            }
            $code = wp_remote_retrieve_response_code($resp);
            if ($code < 200 || $code >= 300) {
                wp_die(__('Unable to retrieve file.', 'secure-pdf'), 502);
            }
            $body = wp_remote_retrieve_body($resp);
            if (!headers_sent()) {
                header('Content-Type: application/pdf');
                header('Content-Disposition: attachment; filename="' . rawurlencode($file_name) . '"');
                header('Content-Length: ' . strlen($body));
            }
            echo $body;
        }
        exit;
    }
    add_action('wp_ajax_secure_pdf_stream', 'secure_pdf_stream_file');
    add_action('wp_ajax_nopriv_secure_pdf_stream', 'secure_pdf_stream_file');

    add_action('wp_ajax_secure_pdf_create_payment_intent', function() {
        if (!isset($_POST['amount'], $_POST['url'])) {
            wp_send_json_error(['message' => 'Missing parameters.'], 400);
        }
        $amount = intval($_POST['amount']);
        if ($amount <= 0) {
            wp_send_json_error(['message' => 'Invalid amount.'], 400);
        }
        $secret = secure_pdf_get_stripe_secret_key();
        if (!$secret) {
            wp_send_json_error(['message' => 'Stripe secret is not configured.'], 500);
        }
        $args = [
            'headers' => [
                'Authorization' => 'Bearer ' . $secret,
            ],
            'body' => [
                'amount' => $amount,
                'currency' => 'usd',
                'automatic_payment_methods[enabled]' => 'true',
                'metadata[url]' => sanitize_text_field($_POST['url']),
            ],
            'timeout' => 30,
        ];
        $resp = wp_remote_post('https://api.stripe.com/v1/payment_intents', $args);
        if (is_wp_error($resp)) {
            wp_send_json_error(['message' => $resp->get_error_message()], 500);
        }
        $code = wp_remote_retrieve_response_code($resp);
        $body = json_decode(wp_remote_retrieve_body($resp), true);
        if ($code >= 200 && $code < 300 && isset($body['client_secret'])) {
            wp_send_json_success(['client_secret' => $body['client_secret']]);
        }
        wp_send_json_error(['message' => 'Stripe error', 'body' => $body], 500);
    });
    add_action('wp_ajax_nopriv_secure_pdf_create_payment_intent', function() {
        do_action('wp_ajax_secure_pdf_create_payment_intent');
    });

    // Registration endpoint (simple minimal flow)
    add_action('wp_ajax_secure_pdf_register', function() {
        if (!isset($_POST['email'], $_POST['password'], $_POST['_wpnonce']) || !wp_verify_nonce($_POST['_wpnonce'], 'secure_pdf')) {
            wp_send_json_error(['message' => 'Invalid request.'], 400);
        }
        $email = sanitize_email($_POST['email']);
        $pass  = (string) $_POST['password'];
        if (!$email || !$pass) {
            wp_send_json_error(['message' => 'Email and password are required.'], 400);
        }
        if (email_exists($email)) {
            // If email exists, attempt to sign user in instead of failing
            $user = get_user_by('email', $email);
            if ($user && wp_check_password($pass, $user->user_pass, $user->ID)) {
                wp_set_current_user($user->ID);
                wp_set_auth_cookie($user->ID, true);
                wp_send_json_success(['logged_in' => true]);
            }
            wp_send_json_error(['message' => 'Account exists. Please sign in with this email.', 'code' => 'account_exists'], 409);
        }
        // Use the email address as the username (allowing '@'), ensure uniqueness
        $username = sanitize_user($email, false);
        if (username_exists($username)) {
            $username = sanitize_user($email . '_' . wp_generate_password(4, false), false);
        }
        $user_id = wp_create_user($username, $pass, $email);
        if (is_wp_error($user_id)) {
            wp_send_json_error(['message' => $user_id->get_error_message()], 500);
        }
        if (isset($_POST['first_name']) && $_POST['first_name'] !== '') {
            $first = sanitize_text_field($_POST['first_name']);
            update_user_meta($user_id, 'first_name', $first);
            // Set display_name to first name
            wp_update_user([ 'ID' => $user_id, 'display_name' => $first ]);
        }
        if (isset($_POST['last_name']) && $_POST['last_name'] !== '') {
            update_user_meta($user_id, 'last_name', sanitize_text_field($_POST['last_name']));
        }
        wp_set_current_user($user_id);
        wp_set_auth_cookie($user_id, true);
        wp_send_json_success(['created' => true, 'user_id' => $user_id]);
    });
    add_action('wp_ajax_nopriv_secure_pdf_register', function() {
        do_action('wp_ajax_secure_pdf_register');
    });

    /** Admin menu & settings page */
    add_action('admin_menu', function () {
        add_submenu_page(
            'flw-plugins',
            'Secure PDF Embedder Settings',
            'Secure PDF Embedder',
            'manage_options',
            'securepdf',
            'secure_pdf_settings_page'
        );
    });

    function secure_pdf_settings_page() {
        $active_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'generator';
        echo '<div class="wrap">';
        echo '<h1>Secure PDF Embedder</h1>';
        echo '<nav class="nav-tab-wrapper">';
        echo '<a href="?page=securepdf&tab=generator" class="nav-tab ' . ($active_tab === 'generator' ? 'nav-tab-active' : '') . '">Generator</a>';
        echo '<a href="?page=securepdf&tab=settings" class="nav-tab ' . ($active_tab === 'settings' ? 'nav-tab-active' : '') . '">Settings</a>';
        echo '</nav>';

        if ($active_tab === 'generator') {
            echo '<h2>Embed Generator</h2>';
            echo '<div class="notice notice-info" style="padding:16px;">';
            echo '<p style="margin-top:0; font-size:14px;"><strong>Secure PDF has moved to the Block Editor.</strong></p>';
            echo '<p style="margin-bottom:12px;">The legacy shortcode generator has been retired. Please use the Gutenberg block for a faster and more modern editing experience.</p>';
            echo '<ul style="list-style:disc; padding-left:20px; margin:0 0 12px 0;">';
            echo '<li>In the editor, click “Add block” and search for <strong>Secure PDF</strong>.</li>';
            echo '<li>Paste your PDF URL and adjust options (Download, Print, Members Only, Mature Content).</li>';
            echo '<li>To convert existing <code>[secure_pdf]</code> shortcodes, use the one-click converter in <a href="?page=securepdf&tab=settings">Settings → Bulk Tools</a>, or open a post and switch the Shortcode block to the Secure PDF block.</li>';
            echo '</ul>';
            echo '<p style="margin-bottom:0;">Tip: Run the bulk converter to update many posts at once.</p>';
            echo '</div>';
        } elseif ($active_tab === 'settings') {
            echo '<h2>Settings</h2>';
            if (isset($_GET['converted'], $_GET['scanned'])) {
                $converted = intval($_GET['converted']);
                $scanned   = intval($_GET['scanned']);
                echo '<div class="notice notice-success" style="padding:12px;">' . sprintf(esc_html__('%d of %d posts updated.', 'secure-pdf'), $converted, $scanned) . '</div>';
            }
            echo '<form method="post" action="options.php">';
            settings_fields('secure_pdf_options_group');
            do_settings_sections('secure-pdf-settings');
            // (settings fields as before...)
            submit_button();
            echo '</form>';

            // Bulk convert tool
            echo '<hr />';
            echo '<h2>Bulk Tools</h2>';
            echo '<p>Convert all existing [secure_pdf] shortcodes in posts/pages to the Secure PDF block.</p>';
            echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
            echo '<input type="hidden" name="action" value="secure_pdf_convert_shortcodes">';
            wp_nonce_field('secure_pdf_convert_shortcodes');
            echo '<button type="submit" class="button button-secondary" onclick="return confirm(\'This will modify post content. Continue?\');">Convert Shortcodes to Blocks</button>';
            echo '</form>';
        }
        echo '</div>';
    }

    add_action('admin_init', function () {
        register_setting('secure_pdf_options_group', 'secure_pdf_join_now_url');
        register_setting('secure_pdf_options_group', 'secure_pdf_button_color');
        register_setting('secure_pdf_options_group', 'secure_pdf_button_color_opacity');
        register_setting('secure_pdf_options_group', 'secure_pdf_button_color_transparent');
        register_setting('secure_pdf_options_group', 'secure_pdf_button_hover_color');
        register_setting('secure_pdf_options_group', 'secure_pdf_button_hover_color_opacity');
        register_setting('secure_pdf_options_group', 'secure_pdf_button_hover_color_transparent');
        register_setting('secure_pdf_options_group', 'secure_pdf_button_font_color');
        register_setting('secure_pdf_options_group', 'secure_pdf_button_font_color_opacity');
        register_setting('secure_pdf_options_group', 'secure_pdf_button_font_color_transparent');
        register_setting('secure_pdf_options_group', 'secure_pdf_footer_background_color');
        register_setting('secure_pdf_options_group', 'secure_pdf_footer_background_color_opacity');
        register_setting('secure_pdf_options_group', 'secure_pdf_footer_background_color_transparent');
        register_setting('secure_pdf_options_group', 'secure_pdf_footer_top_border_color');
        register_setting('secure_pdf_options_group', 'secure_pdf_footer_top_border_color_opacity');
        register_setting('secure_pdf_options_group', 'secure_pdf_footer_top_border_color_transparent');
        register_setting('secure_pdf_options_group', 'secure_pdf_footer_top_border_thickness');
        register_setting('secure_pdf_options_group', 'secure_pdf_footer_top_border_style');
        register_setting('secure_pdf_options_group', 'secure_pdf_viewer_outline_color');
        register_setting('secure_pdf_options_group', 'secure_pdf_viewer_outline_color_opacity');
        register_setting('secure_pdf_options_group', 'secure_pdf_viewer_outline_color_transparent');
        register_setting('secure_pdf_options_group', 'secure_pdf_viewer_border_thickness');
        register_setting('secure_pdf_options_group', 'secure_pdf_viewer_border_style');
        register_setting('secure_pdf_options_group', 'secure_pdf_download_button_text');
        register_setting('secure_pdf_options_group', 'secure_pdf_join_now_button_text');
        register_setting('secure_pdf_options_group', 'secure_pdf_join_button_color');
        register_setting('secure_pdf_options_group', 'secure_pdf_join_button_color_opacity');
        register_setting('secure_pdf_options_group', 'secure_pdf_join_button_color_transparent');
        register_setting('secure_pdf_options_group', 'secure_pdf_join_button_font_color');
        register_setting('secure_pdf_options_group', 'secure_pdf_join_button_font_color_opacity');
        register_setting('secure_pdf_options_group', 'secure_pdf_join_button_font_color_transparent');

        // UI section and fields
        add_settings_section(
            'secure_pdf_style_section',
            __('Style Options', 'secure-pdf'),
            function () {
                echo '<p>' . esc_html__('Configure global styles for the Secure PDF Embedder.', 'secure-pdf') . '</p>';
            },
            'secure-pdf-settings'
        );

        $color_control = function ($base, $default_hex, $label = '') {
            $hex  = esc_attr(get_option($base, $default_hex));
            $op   = esc_attr(get_option($base . '_opacity', '100'));
            $tran = get_option($base . '_transparent', '0') === '1' ? 'checked' : '';
            echo '<div class="secure-pdf-color-control" style="display:flex;align-items:center;gap:8px;flex-wrap:wrap;">';
            echo '<input type="text" class="regular-text secure-pdf-color-picker" name="' . esc_attr($base) . '" value="' . $hex . '" data-default-color="' . esc_attr($default_hex) . '" placeholder="#rrggbb" />';
            echo '<label style="display:flex;align-items:center;gap:6px;">';
            echo '<input type="checkbox" class="secure-pdf-transparent" name="' . esc_attr($base) . '_transparent" value="1" ' . $tran . ' /> ' . esc_html__('Transparent', 'secure-pdf');
            echo '</label>';
            echo '<label style="display:flex;align-items:center;gap:6px;">' . esc_html__('Opacity', 'secure-pdf') . ' '; 
            echo '<input type="range" class="secure-pdf-opacity" name="' . esc_attr($base) . '_opacity" min="0" max="100" step="1" value="' . $op . '" />';
            echo ' <span class="secure-pdf-opacity-value">' . intval($op) . '%</span>';
            echo '</label>';
            if (!empty($label)) echo '<p class="description" style="width:100%">' . esc_html($label) . '</p>';
            echo '</div>';
        };
        $text_field = function ($option, $placeholder = '', $default = '') {
            $val = esc_attr(get_option($option, $default));
            echo '<input type="text" class="regular-text" name="' . esc_attr($option) . '" value="' . $val . '" placeholder="' . esc_attr($placeholder) . '" />';
        };
        $number_field = function ($option, $min = 0, $default = '') {
            $val = esc_attr(get_option($option, $default));
            echo '<input type="number" class="small-text" min="' . intval($min) . '" name="' . esc_attr($option) . '" value="' . $val . '" />';
        };
        $border_style_field = function ($option, $default = 'solid') {
            $current = get_option($option, $default);
            $styles = ['none','solid','dashed','dotted','double','groove','ridge','inset','outset'];
            echo '<select name="' . esc_attr($option) . '">';
            foreach ($styles as $style) {
                echo '<option value="' . esc_attr($style) . '"' . selected($current, $style, false) . '>' . esc_html($style) . '</option>';
            }
            echo '</select>';
        };

        // Button styles
        add_settings_field('secure_pdf_button_color', __('Button Background', 'secure-pdf'), function () use ($color_control) { $color_control('secure_pdf_button_color', '#0073aa', __('Default #0073aa', 'secure-pdf')); }, 'secure-pdf-settings', 'secure_pdf_style_section');
        add_settings_field('secure_pdf_button_hover_color', __('Button Hover Background', 'secure-pdf'), function () use ($color_control) { $color_control('secure_pdf_button_hover_color', '#005177', __('Default #005177', 'secure-pdf')); }, 'secure-pdf-settings', 'secure_pdf_style_section');
        add_settings_field('secure_pdf_button_font_color', __('Button Text Color', 'secure-pdf'), function () use ($color_control) { $color_control('secure_pdf_button_font_color', '#ffffff', __('Default #ffffff', 'secure-pdf')); }, 'secure-pdf-settings', 'secure_pdf_style_section');

        // Footer styles
        add_settings_field('secure_pdf_footer_background_color', __('Footer Background', 'secure-pdf'), function () use ($color_control) { $color_control('secure_pdf_footer_background_color', '#333333', __('Default #333333', 'secure-pdf')); }, 'secure-pdf-settings', 'secure_pdf_style_section');
        add_settings_field('secure_pdf_footer_top_border_color', __('Footer Top Border Color', 'secure-pdf'), function () use ($color_control) { $color_control('secure_pdf_footer_top_border_color', '#000000', __('Default #000000', 'secure-pdf')); }, 'secure-pdf-settings', 'secure_pdf_style_section');
        add_settings_field('secure_pdf_footer_top_border_thickness', __('Footer Top Border Thickness (px)', 'secure-pdf'), function () use ($number_field) { $number_field('secure_pdf_footer_top_border_thickness', 0, '4'); }, 'secure-pdf-settings', 'secure_pdf_style_section');
        add_settings_field('secure_pdf_footer_top_border_style', __('Footer Top Border Style', 'secure-pdf'), function () use ($border_style_field) { $border_style_field('secure_pdf_footer_top_border_style', 'solid'); }, 'secure-pdf-settings', 'secure_pdf_style_section');

        // Viewer border
        add_settings_field('secure_pdf_viewer_outline_color', __('Viewer Border Color', 'secure-pdf'), function () use ($color_control) { $color_control('secure_pdf_viewer_outline_color', '#cccccc', __('Default #cccccc', 'secure-pdf')); }, 'secure-pdf-settings', 'secure_pdf_style_section');
        add_settings_field('secure_pdf_viewer_border_thickness', __('Viewer Border Thickness (px)', 'secure-pdf'), function () use ($number_field) { $number_field('secure_pdf_viewer_border_thickness', 0, '1'); }, 'secure-pdf-settings', 'secure_pdf_style_section');
        add_settings_field('secure_pdf_viewer_border_style', __('Viewer Border Style', 'secure-pdf'), function () use ($border_style_field) { $border_style_field('secure_pdf_viewer_border_style', 'solid'); }, 'secure-pdf-settings', 'secure_pdf_style_section');

        // Texts and links
        add_settings_field('secure_pdf_download_button_text', __('Download Button Text', 'secure-pdf'), function () use ($text_field) { $text_field('secure_pdf_download_button_text', 'Download', 'Download'); }, 'secure-pdf-settings', 'secure_pdf_style_section');
        add_settings_field('secure_pdf_join_now_url', __('Join Now URL', 'secure-pdf'), function () use ($text_field) { $text_field('secure_pdf_join_now_url', 'https://example.com/join', '#'); }, 'secure-pdf-settings', 'secure_pdf_style_section');
        add_settings_field('secure_pdf_join_now_button_text', __('Join Now Button Text', 'secure-pdf'), function () use ($text_field) { $text_field('secure_pdf_join_now_button_text', 'Join Now', 'Join Now'); }, 'secure-pdf-settings', 'secure_pdf_style_section');
        add_settings_field('secure_pdf_join_button_color', __('Join Button Background', 'secure-pdf'), function () use ($color_control) { $color_control('secure_pdf_join_button_color', '#d63638', __('Default #d63638', 'secure-pdf')); }, 'secure-pdf-settings', 'secure_pdf_style_section');
        add_settings_field('secure_pdf_join_button_font_color', __('Join Button Text Color', 'secure-pdf'), function () use ($color_control) { $color_control('secure_pdf_join_button_font_color', '#ffffff', __('Default #ffffff', 'secure-pdf')); }, 'secure-pdf-settings', 'secure_pdf_style_section');

        // Open button settings (cover overlay)
        register_setting('secure_pdf_options_group', 'secure_pdf_open_button_text');
        add_settings_field('secure_pdf_open_button_text', __('Open Button Text', 'secure-pdf'), function () use ($text_field) { $text_field('secure_pdf_open_button_text', __('Open Document', 'secure-pdf'), ''); }, 'secure-pdf-settings', 'secure_pdf_style_section');

        register_setting('secure_pdf_options_group', 'secure_pdf_open_button_color');
        register_setting('secure_pdf_options_group', 'secure_pdf_open_button_color_opacity');
        register_setting('secure_pdf_options_group', 'secure_pdf_open_button_color_transparent');
        add_settings_field('secure_pdf_open_button_color', __('Open Button Background', 'secure-pdf'), function () use ($color_control) { $color_control('secure_pdf_open_button_color', '#0073aa', __('Default #0073aa', 'secure-pdf')); }, 'secure-pdf-settings', 'secure_pdf_style_section');

        register_setting('secure_pdf_options_group', 'secure_pdf_open_button_hover_color');
        register_setting('secure_pdf_options_group', 'secure_pdf_open_button_hover_color_opacity');
        register_setting('secure_pdf_options_group', 'secure_pdf_open_button_hover_color_transparent');
        add_settings_field('secure_pdf_open_button_hover_color', __('Open Button Hover Background', 'secure-pdf'), function () use ($color_control) { $color_control('secure_pdf_open_button_hover_color', '#005177', __('Default #005177', 'secure-pdf')); }, 'secure-pdf-settings', 'secure_pdf_style_section');

        register_setting('secure_pdf_options_group', 'secure_pdf_open_button_font_color');
        register_setting('secure_pdf_options_group', 'secure_pdf_open_button_font_color_opacity');
        register_setting('secure_pdf_options_group', 'secure_pdf_open_button_font_color_transparent');
        add_settings_field('secure_pdf_open_button_font_color', __('Open Button Text Color', 'secure-pdf'), function () use ($color_control) { $color_control('secure_pdf_open_button_font_color', '#ffffff', __('Default #ffffff', 'secure-pdf')); }, 'secure-pdf-settings', 'secure_pdf_style_section');
    });

    /** Bulk convert shortcodes to blocks */
    add_action('admin_post_secure_pdf_convert_shortcodes', function () {
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have permission to perform this action.', 'secure-pdf'));
        }
        check_admin_referer('secure_pdf_convert_shortcodes');

        $post_types = get_post_types(['public' => true], 'names');
        $query = new WP_Query([
            'post_type' => array_values($post_types),
            'post_status' => ['publish', 'draft', 'future', 'private'],
            's' => '[secure_pdf',
            'posts_per_page' => -1,
            'fields' => 'ids',
            'no_found_rows' => true,
            'ignore_sticky_posts' => true,
        ]);

        $converted = 0;
        $scanned = 0;
        if ($query->have_posts()) {
            foreach ($query->posts as $post_id) {
                $content = get_post_field('post_content', $post_id);
                $scanned++;
                $new_content = secure_pdf_convert_shortcodes_in_content($content);
                if ($new_content !== $content) {
                    wp_update_post([
                        'ID' => $post_id,
                        'post_content' => $new_content,
                    ]);
                    $converted++;
                }
            }
        }

        $redirect = add_query_arg([
            'page' => 'securepdf',
            'tab' => 'settings',
            'converted' => $converted,
            'scanned' => $scanned,
        ], admin_url('admin.php'));
        wp_safe_redirect($redirect);
        exit;
    });

    /** Helper: convert [secure_pdf] shortcodes to block comments in content */
    function secure_pdf_convert_shortcodes_in_content($content) {
        if (strpos($content, 'secure_pdf') === false) {
            return $content;
        }

        // Prefer robust block parsing when available
        if (function_exists('parse_blocks') && function_exists('serialize_blocks')) {
            $blocks = parse_blocks($content);

            $map_attrs = function(array $attrs): array {
                $map = [
                    'url' => '',
                    'width' => '100%',
                    'height' => '600px',
                    'show_download' => 'false',
                    'show_print' => 'false',
                    'members_only' => 'false',
                    'mature_content' => 'false',
                    'cover_image' => '',
                    'open_button_text' => '',
                    'requires_purchase' => 'false',
                    'price' => '',
                ];
                foreach ($map as $key => $def) {
                    if (isset($attrs[$key])) {
                        $val = (string) $attrs[$key];
                        if (in_array($key, ['show_download','show_print','members_only','mature_content'], true)) {
                            $val = ($val === 'true') ? 'true' : 'false';
                        }
                        $map[$key] = $val;
                    }
                }
                return $map;
            };

            $transform = function(array $blocks) use (&$transform, $map_attrs) {
                $out = [];
                foreach ($blocks as $b) {
                    // Recurse first
                    if (!empty($b['innerBlocks'])) {
                        $b['innerBlocks'] = $transform($b['innerBlocks']);
                    }

                    if (isset($b['blockName']) && $b['blockName'] === 'core/shortcode') {
                        $raw = isset($b['innerHTML']) ? trim($b['innerHTML']) : '';
                        // Strip wrapping tags e.g., <p>shortcode</p>
                        $raw_text = trim(wp_strip_all_tags($raw));
                        if (preg_match('/\[\s*secure_pdf\s*(.*?)\s*\]/i', $raw_text, $m)) {
                            $attrs = shortcode_parse_atts($m[1]);
                            if (!is_array($attrs)) { $attrs = []; }
                            $mapped = $map_attrs($attrs);
                            $out[] = [
                                'blockName' => 'flw/secure-pdf',
                                'attrs' => $mapped,
                                'innerBlocks' => [],
                                'innerHTML' => '',
                                'innerContent' => [],
                            ];
                            continue;
                        }
                    }

                    // Also convert plain shortcodes in null blocks or paragraphs containing only the shortcode
                    if ((empty($b['blockName']) || $b['blockName'] === 'core/paragraph') && !empty($b['innerHTML'])) {
                        $raw_text = trim(wp_strip_all_tags($b['innerHTML']));
                        if (preg_match('/^\[\s*secure_pdf\s*(.*?)\s*\]$/i', $raw_text, $m)) {
                            $attrs = shortcode_parse_atts($m[1]);
                            if (!is_array($attrs)) { $attrs = []; }
                            $mapped = $map_attrs($attrs);
                            $out[] = [
                                'blockName' => 'flw/secure-pdf',
                                'attrs' => $mapped,
                                'innerBlocks' => [],
                                'innerHTML' => '',
                                'innerContent' => [],
                            ];
                            continue;
                        }
                    }

                    $out[] = $b;
                }
                return $out;
            };

            $converted_blocks = $transform($blocks);
            return serialize_blocks($converted_blocks);
        }

        // Fallback regex approach for older WP: replace shortcode blocks (relaxed) and raw shortcodes
        $new_content = $content;
        $block_wrapper_pattern = '/<!--\s*wp:shortcode\s*-->[\s\S]*?<!--\s*\/wp:shortcode\s*-->/i';
        $new_content = preg_replace_callback($block_wrapper_pattern, function ($m) {
            $segment = $m[0];
            // Find shortcode inside possibly wrapped HTML
            $inner = trim(preg_replace('/^<!--\s*wp:shortcode\s*-->|<!--\s*\/wp:shortcode\s*-->$/i', '', $segment));
            $inner = wp_strip_all_tags($inner);
            if (preg_match('/\[\s*secure_pdf\s*(.*?)\s*\]/i', $inner, $am)) {
                $attrs = shortcode_parse_atts($am[1]);
                if (!is_array($attrs)) $attrs = [];
                $map = [
                    'url' => '', 'width' => '100%', 'height' => '600px',
                    'show_download' => 'false', 'show_print' => 'false',
                    'members_only' => 'false', 'mature_content' => 'false',
                    'cover_image' => '', 'open_button_text' => '', 'requires_purchase' => 'false', 'price' => '',
                ];
                foreach ($map as $key => $def) {
                    if (isset($attrs[$key])) {
                        $val = (string) $attrs[$key];
                        if (in_array($key, ['show_download','show_print','members_only','mature_content'], true)) {
                            $val = ($val === 'true') ? 'true' : 'false';
                        }
                        $map[$key] = $val;
                    }
                }
                $json = wp_json_encode($map);
                return "<!-- wp:flw/secure-pdf {$json} /-->";
            }
            return $segment; // leave untouched if not our shortcode
        }, $new_content);

        $pattern = get_shortcode_regex(['secure_pdf']);
        if ($pattern && preg_match_all('/' . $pattern . '/s', $new_content, $matches, PREG_SET_ORDER)) {
            foreach ($matches as $shortcode) {
                $tag = isset($shortcode[2]) ? $shortcode[2] : '';
                if (strtolower($tag) !== 'secure_pdf') continue;
                $attrs_raw = isset($shortcode[3]) ? trim($shortcode[3]) : '';
                $attrs = shortcode_parse_atts($attrs_raw);
                if (!is_array($attrs)) $attrs = [];
                $map = [
                    'url' => '', 'width' => '100%', 'height' => '600px',
                    'show_download' => 'false', 'show_print' => 'false',
                    'members_only' => 'false', 'mature_content' => 'false',
                    'cover_image' => '', 'open_button_text' => '', 'requires_purchase' => 'false', 'price' => '',
                ];
                foreach ($map as $key => $def) {
                    if (isset($attrs[$key])) {
                        $val = (string) $attrs[$key];
                        if (in_array($key, ['show_download','show_print','members_only','mature_content'], true)) {
                            $val = ($val === 'true') ? 'true' : 'false';
                        }
                        $map[$key] = $val;
                    }
                }
                $json = wp_json_encode($map);
                $block = "<!-- wp:flw/secure-pdf {$json} /-->";
                $new_content = str_replace($shortcode[0], $block, $new_content);
            }
        }

        return $new_content;
    }
