<?php
/*
Plugin Name: Remote Uploader
Description: Upload media to download host while preserving Elementor paths + transfer files from main host to download host
Version: 2.1.4.2
Author: Tyson Brooks
*/

defined('ABSPATH') || exit;

// Determine plugin slug from directory name of this file
$wda_plugin_slug = basename(dirname(__FILE__));
// Load translations
add_action('init', function () use ($wda_plugin_slug) {
	load_plugin_textdomain($wda_plugin_slug, false, $wda_plugin_slug . '/languages');
});

// Updater bootstrap (requires updater.php at plugin root)
require_once __DIR__ . '/updater.php';
FLW_Starter_Update::boot(__FILE__, $wda_plugin_slug, 'https://frostlineworks.com/updates/');
/**
 * CONFIGURATION
 * You can define constants in wp-config.php to override UI settings.
 * If not defined, the Settings UI values (or internal defaults) are used.
 */

add_filter('upload_dir', 'remote_uploader_custom_upload_dir');
function remote_uploader_custom_upload_dir($dirs) {
    $path = $dirs['path'];
    $subdir = $dirs['subdir'];

    if (strpos($path, '/uploads/elementor/') !== false) {
        return $dirs;
    }

    $backtrace = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS);
    foreach ($backtrace as $call) {
        if (!empty($call['class']) && strpos($call['class'], 'Elementor\\Core\\Files\\CSS') === 0) {
            return $dirs;
        }
    }
    
    // Do not override base URL globally; URLs are handled per-attachment for offloaded files.

    return $dirs;
}

add_filter('wp_handle_upload', 'remote_uploader_move_to_ftp');
function remote_uploader_move_to_ftp($upload) {
    // Defer all offloading until after WordPress generates attachment metadata and sizes.
    // The actual offload and local deletions happen in remote_uploader_offload_after_metadata().
    return $upload;
}

// Reusable transfer helper to send a local file to remote path based on settings
function remote_uploader_transfer_local_to_remote($relative_path) {
    if (function_exists('set_time_limit')) { @set_time_limit(0); }
    $local = WP_CONTENT_DIR . '/uploads/' . ltrim($relative_path, '/');
    if (!file_exists($local)) {
        error_log('❌ [Remote Uploader] Local file not found: ' . $local);
        return false;
    }
    $remote_file = rtrim(remote_uploader_get_setting('ftp_basedir'), '/') . '/' . ltrim($relative_path, '/');

    $protocol = strtolower(remote_uploader_get_setting('protocol'));
    $host = remote_uploader_get_setting('ftp_host');
    $port = intval(remote_uploader_get_setting('port'));
    if ($port <= 0) {
        $port = ($protocol === 'sftp') ? 22 : 21;
    }

    if ($protocol === 'sftp') {
        if (!function_exists('ssh2_connect')) {
            error_log('❌ [Remote Uploader] SFTP selected but ext-ssh2 is not installed.');
            return false;
        }
        $conn = @ssh2_connect($host, $port);
        if (!$conn) {
            error_log('❌ [Remote Uploader] SFTP connect failed to ' . $host . ':' . $port);
            return false;
        }
        $user = remote_uploader_get_setting('ftp_username');
        $pass = remote_uploader_get_setting('ftp_password');
        if (!@ssh2_auth_password($conn, $user, $pass)) {
            error_log('❌ [Remote Uploader] SFTP auth failed for user ' . $user . ' @ ' . $host);
            return false;
        }
        $sftp = @ssh2_sftp($conn);
        if (!$sftp) {
            error_log('❌ [Remote Uploader] SFTP subsystem initialization failed');
            return false;
        }
        $dir = dirname($remote_file);
        if (!remote_uploader_sftp_ensure_dir($sftp, $dir)) {
            error_log('❌ [Remote Uploader] SFTP ensure-dir failed for: ' . $dir);
            return false;
        }
        $remoteStream = @fopen('ssh2.sftp://' . $sftp . $remote_file, 'w');
        if (!$remoteStream) {
            error_log('❌ [Remote Uploader] SFTP fopen remote failed: ' . $remote_file);
            return false;
        }
        $localStream = @fopen($local, 'r');
        if (!$localStream) {
            fclose($remoteStream);
            error_log('❌ [Remote Uploader] fopen local failed: ' . $local);
            return false;
        }
        $bytes = @stream_copy_to_stream($localStream, $remoteStream);
        fclose($localStream);
        fclose($remoteStream);
        if ($bytes === false) {
            error_log('❌ [Remote Uploader] SFTP stream copy failed for: ' . basename($local));
            return false;
        }
        return true;
    }

    // FTP
    $ftp_conn = @ftp_connect($host, $port);
    if (!$ftp_conn) {
        error_log('❌ [Remote Uploader] FTP connect failed to ' . $host . ':' . $port);
        return false;
    }
    $login = @ftp_login($ftp_conn, remote_uploader_get_setting('ftp_username'), remote_uploader_get_setting('ftp_password'));
    if (!$login) {
        error_log('❌ [Remote Uploader] FTP login failed for user ' . remote_uploader_get_setting('ftp_username') . ' @ ' . $host);
        ftp_close($ftp_conn);
        return false;
    }
    @ftp_pasv($ftp_conn, true);
    @ftp_set_option($ftp_conn, FTP_TIMEOUT_SEC, 300);
    // Ensure directories
    $dirs = explode('/', dirname($remote_file));
    $path = '';
    foreach ($dirs as $dir) {
        if ($dir === '') continue;
        $path .= '/' . $dir;
        @ftp_mkdir($ftp_conn, $path);
    }
    $ok = @ftp_put($ftp_conn, $remote_file, $local, FTP_BINARY);
    if (!$ok) {
        error_log('❌ [Remote Uploader] FTP put failed to: ' . $remote_file . ' from local: ' . $local);
    }
    ftp_close($ftp_conn);
    return $ok;
}

// After WP generates attachment metadata, offload original and all sizes, then delete locals
add_filter('wp_generate_attachment_metadata', 'remote_uploader_offload_after_metadata', 20, 2);
function remote_uploader_offload_after_metadata($metadata, $attachment_id) {
    $relative = get_post_meta($attachment_id, '_wp_attached_file', true);
    if (!$relative) return $metadata;
    if (!remote_uploader_is_extension_allowed($relative)) return $metadata;

    $files = [];
    $files[] = $relative;
    $base_dir = trailingslashit(dirname($relative));
    if (is_array($metadata) && !empty($metadata['sizes']) && is_array($metadata['sizes'])) {
        foreach ($metadata['sizes'] as $size) {
            if (!empty($size['file'])) {
                $files[] = $base_dir . $size['file'];
            }
        }
    }

    $all_ok = true;
    foreach ($files as $rel) {
        $ok = remote_uploader_transfer_local_to_remote($rel);
        if ($ok) {
            @unlink(WP_CONTENT_DIR . '/uploads/' . ltrim($rel, '/'));
        } else {
            $all_ok = false;
            error_log('❌ [Remote Uploader] Failed to offload (post-meta stage): ' . $rel);
        }
    }

    if ($all_ok) {
        update_post_meta($attachment_id, '_remote_uploader_offloaded', '1');
        remote_uploader_mark_offloaded_by_path($relative);
    }

    return $metadata;
}

add_action('admin_init', 'remote_uploader_register_settings');
function remote_uploader_register_settings() {
    register_setting('remote_uploader_settings', 'remoteuploader_subdomain_url', [
        'type' => 'string',
        'sanitize_callback' => 'esc_url_raw',
        'show_in_rest' => false,
    ]);
    register_setting('remote_uploader_settings', 'remoteuploader_ftp_host', [
        'type' => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'show_in_rest' => false,
    ]);
    register_setting('remote_uploader_settings', 'remoteuploader_ftp_username', [
        'type' => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'show_in_rest' => false,
    ]);
    register_setting('remote_uploader_settings', 'remoteuploader_ftp_password', [
        'type' => 'string',
        'sanitize_callback' => 'remote_uploader_sanitize_password',
        'show_in_rest' => false,
    ]);
    register_setting('remote_uploader_settings', 'remoteuploader_ftp_basedir', [
        'type' => 'string',
        'sanitize_callback' => 'remote_uploader_sanitize_path',
        'show_in_rest' => false,
    ]);
    // New: protocol and port
    register_setting('remote_uploader_settings', 'remoteuploader_protocol', [
        'type' => 'string',
        'sanitize_callback' => 'remote_uploader_sanitize_protocol',
        'show_in_rest' => false,
    ]);
    register_setting('remote_uploader_settings', 'remoteuploader_port', [
        'type' => 'integer',
        'sanitize_callback' => 'remote_uploader_sanitize_port',
        'show_in_rest' => false,
    ]);
    // New: allowed extensions (comma-separated, no dots)
    register_setting('remote_uploader_settings', 'remoteuploader_extensions', [
        'type' => 'string',
        'sanitize_callback' => 'remote_uploader_sanitize_extensions',
        'show_in_rest' => false,
    ]);

    add_settings_section('remote_uploader_main', 'Remote Uploader Settings', function () {
        echo '<p>Configure the download host URL and connection used to offload media uploads.</p>';
    }, 'remote_uploader');

    add_settings_field('remoteuploader_subdomain_url', 'Download Host URL', 'remote_uploader_field_subdomain', 'remote_uploader', 'remote_uploader_main');
    add_settings_field('remoteuploader_protocol', 'Protocol', 'remote_uploader_field_protocol', 'remote_uploader', 'remote_uploader_main');
    add_settings_field('remoteuploader_port', 'Port', 'remote_uploader_field_port', 'remote_uploader', 'remote_uploader_main');
    add_settings_field('remoteuploader_ftp_host', 'Host', 'remote_uploader_field_ftp_host', 'remote_uploader', 'remote_uploader_main');
    add_settings_field('remoteuploader_ftp_username', 'Username', 'remote_uploader_field_ftp_username', 'remote_uploader', 'remote_uploader_main');
    add_settings_field('remoteuploader_ftp_password', 'Password', 'remote_uploader_field_ftp_password', 'remote_uploader', 'remote_uploader_main');
    add_settings_field('remoteuploader_ftp_basedir', 'Base Directory', 'remote_uploader_field_ftp_basedir', 'remote_uploader', 'remote_uploader_main');
    add_settings_field('remoteuploader_extensions', 'Allowed Extensions', 'remote_uploader_field_extensions', 'remote_uploader', 'remote_uploader_main');
}

function remote_uploader_sanitize_password($value) {
    return is_string($value) ? $value : '';
}

function remote_uploader_sanitize_path($value) {
    $value = is_string($value) ? trim($value) : '';
    $value = preg_replace('#/+#', '/', $value);
    return $value;
}

function remote_uploader_sanitize_protocol($value) {
    $value = is_string($value) ? strtolower(trim($value)) : 'ftp';
    return in_array($value, ['ftp', 'sftp'], true) ? $value : 'ftp';
}

function remote_uploader_sanitize_port($value) {
    $value = intval($value);
    if ($value <= 0 || $value > 65535) {
        return 21;
    }
    return $value;
}

function remote_uploader_sanitize_extensions($value) {
    if (!is_string($value)) return '';
    $parts = array_filter(array_map('trim', explode(',', strtolower($value))));
    $clean = [];
    foreach ($parts as $p) {
        $p = ltrim($p, '.');
        if ($p !== '' && preg_match('/^[a-z0-9]+$/', $p)) {
            $clean[] = $p;
        }
    }
    $clean = array_unique($clean);
    return implode(',', $clean);
}

add_action('admin_menu', 'remote_uploader_add_admin_menu');
function remote_uploader_add_admin_menu() {
    add_options_page('Remote Uploader', 'Remote Uploader', 'manage_options', 'remote_uploader', 'remote_uploader_render_settings_page');
}

function remote_uploader_render_settings_page() {
    if (!current_user_can('manage_options')) {
        return;
    }
    echo '<div class="wrap">';
    echo '<h1>Remote Uploader</h1>';
    remote_uploader_render_constants_notice();
    echo '<form method="post" action="options.php">';
    settings_fields('remote_uploader_settings');
    do_settings_sections('remote_uploader');
    submit_button();
    echo '</form>';
    echo '</div>';
}

function remote_uploader_render_constants_notice() {
    $overrides = [];
    if (defined('REMOTEUPLOADER_SUBDOMAIN_URL')) $overrides[] = 'Download Host URL';
    if (defined('REMOTEUPLOADER_PROTOCOL')) $overrides[] = 'Protocol';
    if (defined('REMOTEUPLOADER_PORT')) $overrides[] = 'Port';
    if (defined('REMOTEUPLOADER_FTP_HOST')) $overrides[] = 'Host';
    if (defined('REMOTEUPLOADER_FTP_USERNAME')) $overrides[] = 'Username';
    if (defined('REMOTEUPLOADER_FTP_PASSWORD')) $overrides[] = 'Password';
    if (defined('REMOTEUPLOADER_FTP_BASEDIR')) $overrides[] = 'Base Directory';
    if (defined('REMOTEUPLOADER_EXTENSIONS')) $overrides[] = 'Allowed Extensions';
    if (!empty($overrides)) {
        $fields = implode(', ', $overrides);
        echo '<div class="notice notice-warning"><p><strong>Note:</strong> The following settings are currently overridden by constants in <code>wp-config.php</code>: ' . esc_html($fields) . '. Remove those constants to manage them here.</p></div>';
    }
}

function remote_uploader_field_subdomain() {
    $value = esc_attr(remote_uploader_get_setting('subdomain_url'));
    echo '<input type="url" class="regular-text" name="remoteuploader_subdomain_url" value="' . $value . '" placeholder="https://dl.example.com/uploads">';
}

function remote_uploader_field_protocol() {
    $value = esc_attr(remote_uploader_get_setting('protocol'));
    echo '<select name="remoteuploader_protocol">';
    echo '<option value="ftp"' . selected($value, 'ftp', false) . '>FTP</option>';
    echo '<option value="sftp"' . selected($value, 'sftp', false) . '>SFTP</option>';
    echo '</select>';
}

function remote_uploader_field_port() {
    $value = esc_attr(remote_uploader_get_setting('port'));
    echo '<input type="number" class="small-text" name="remoteuploader_port" value="' . $value . '" min="1" max="65535"> <span class="description">FTP default 21, SFTP default 22</span>';
}

function remote_uploader_field_ftp_host() {
    $value = esc_attr(remote_uploader_get_setting('ftp_host'));
    echo '<input type="text" class="regular-text" name="remoteuploader_ftp_host" value="' . $value . '" placeholder="ftp.example.com">';
}

function remote_uploader_field_ftp_username() {
    $value = esc_attr(remote_uploader_get_setting('ftp_username'));
    echo '<input type="text" class="regular-text" name="remoteuploader_ftp_username" value="' . $value . '" placeholder="ftpuser">';
}

function remote_uploader_field_ftp_password() {
    $value = esc_attr(remote_uploader_get_setting('ftp_password'));
    echo '<input type="password" class="regular-text" name="remoteuploader_ftp_password" value="' . $value . '" autocomplete="new-password">';
}

function remote_uploader_field_ftp_basedir() {
    $value = esc_attr(remote_uploader_get_setting('ftp_basedir'));
    echo '<input type="text" class="regular-text code" name="remoteuploader_ftp_basedir" value="' . $value . '" placeholder="/domains/example.com/public_html/uploads">';
}

function remote_uploader_field_extensions() {
    $value = esc_attr(remote_uploader_get_setting('extensions'));
    echo '<input type="text" class="regular-text code" name="remoteuploader_extensions" value="' . $value . '" placeholder="jpg,jpeg,png,webp,svg,mp4">';
    echo '<p class="description">Comma-separated list without dots. Only these extensions will be offloaded to the remote server.</p>';
}

function remote_uploader_get_setting($key) {
    $defaults = [
        'subdomain_url' => defined('REMOTEUPLOADER_SUBDOMAIN_URL') ? REMOTEUPLOADER_SUBDOMAIN_URL : 'https://dl.site.com/uploads',
        'protocol' => defined('REMOTEUPLOADER_PROTOCOL') ? REMOTEUPLOADER_PROTOCOL : 'ftp',
        'port' => defined('REMOTEUPLOADER_PORT') ? REMOTEUPLOADER_PORT : 21,
        'ftp_host' => defined('REMOTEUPLOADER_FTP_HOST') ? REMOTEUPLOADER_FTP_HOST : 'ftp.site.com',
        'ftp_username' => defined('REMOTEUPLOADER_FTP_USERNAME') ? REMOTEUPLOADER_FTP_USERNAME : 'user',
        'ftp_password' => defined('REMOTEUPLOADER_FTP_PASSWORD') ? REMOTEUPLOADER_FTP_PASSWORD : 'pass',
        'ftp_basedir' => defined('REMOTEUPLOADER_FTP_BASEDIR') ? REMOTEUPLOADER_FTP_BASEDIR : '/domains/site.com/public_html/uploads',
        'extensions' => defined('REMOTEUPLOADER_EXTENSIONS') ? REMOTEUPLOADER_EXTENSIONS : '',
    ];

    $map = [
        'subdomain_url' => 'remoteuploader_subdomain_url',
        'protocol' => 'remoteuploader_protocol',
        'port' => 'remoteuploader_port',
        'ftp_host' => 'remoteuploader_ftp_host',
        'ftp_username' => 'remoteuploader_ftp_username',
        'ftp_password' => 'remoteuploader_ftp_password',
        'ftp_basedir' => 'remoteuploader_ftp_basedir',
        'extensions' => 'remoteuploader_extensions',
    ];

    $option_key = isset($map[$key]) ? $map[$key] : null;
    if (!$option_key) return isset($defaults[$key]) ? $defaults[$key] : '';

    // If a constant is defined, prefer it (explicit override)
    if (
        ($key === 'subdomain_url' && defined('REMOTEUPLOADER_SUBDOMAIN_URL')) ||
        ($key === 'protocol' && defined('REMOTEUPLOADER_PROTOCOL')) ||
        ($key === 'port' && defined('REMOTEUPLOADER_PORT')) ||
        ($key === 'ftp_host' && defined('REMOTEUPLOADER_FTP_HOST')) ||
        ($key === 'ftp_username' && defined('REMOTEUPLOADER_FTP_USERNAME')) ||
        ($key === 'ftp_password' && defined('REMOTEUPLOADER_FTP_PASSWORD')) ||
        ($key === 'ftp_basedir' && defined('REMOTEUPLOADER_FTP_BASEDIR')) ||
        ($key === 'extensions' && defined('REMOTEUPLOADER_EXTENSIONS'))
    ) {
        return $defaults[$key];
    }

    $value = get_option($option_key);
    if ($value === false || $value === '' || $value === null) {
        return $defaults[$key];
    }
    return $value;
}

// Enforce allowed extensions: only offload when extension is in allowlist
function remote_uploader_is_extension_allowed($file_path) {
    $list = remote_uploader_get_setting('extensions');
    if ($list === '') return false;
    $allowed = array_filter(array_map('trim', explode(',', strtolower($list))));
    $ext = strtolower(pathinfo($file_path, PATHINFO_EXTENSION));
    return $ext !== '' && in_array($ext, $allowed, true);
}

function remote_uploader_sftp_stat($sftp, $path) {
    if (!is_resource($sftp)) return false;
    return @ssh2_sftp_stat($sftp, $path);
}

function remote_uploader_sftp_dir_exists($sftp, $path) {
    $st = remote_uploader_sftp_stat($sftp, $path);
    return is_array($st) && isset($st['mode']) && ($st['mode'] & 040000) === 040000;
}

function remote_uploader_sftp_file_exists($sftp, $path) {
    $st = remote_uploader_sftp_stat($sftp, $path);
    return is_array($st) && isset($st['mode']) && ($st['mode'] & 0100000) === 0100000;
}

function remote_uploader_sftp_ensure_dir($sftp, $dir_path) {
    $parts = explode('/', $dir_path);
    $build = '';
    foreach ($parts as $seg) {
        if ($seg === '') continue;
        $build .= '/' . $seg;
        if (!remote_uploader_sftp_dir_exists($sftp, $build)) {
            if (!@ssh2_sftp_mkdir($sftp, $build, 0775)) {
                // If still not present, log warning
                if (!remote_uploader_sftp_dir_exists($sftp, $build)) {
                    error_log('❌ [Remote Uploader] SFTP mkdir failed for: ' . $build);
                    return false;
                }
            }
        }
    }
    return true;
}

// Per-attachment URL filter: point to CDN when local file is missing
add_filter('wp_get_attachment_url', 'remote_uploader_filter_attachment_url', 10, 2);
function remote_uploader_filter_attachment_url($url, $post_id) {
    $relative = get_post_meta($post_id, '_wp_attached_file', true);
    if (!$relative) return $url;
    $local = WP_CONTENT_DIR . '/uploads/' . ltrim($relative, '/');
    if (file_exists($local)) return $url;
    $cdn = rtrim(remote_uploader_get_setting('subdomain_url'), '/');
    if ($cdn === '') return $url;
    return $cdn . '/' . ltrim($relative, '/');
}

add_action('delete_attachment', 'remote_uploader_delete_remote_files');
function remote_uploader_delete_remote_files($post_id) {
    $relative_path = get_post_meta($post_id, '_wp_attached_file', true);
    if (empty($relative_path) || !is_string($relative_path)) {
        return;
    }

    $files = [];
    $files[] = $relative_path;

    $meta = function_exists('wp_get_attachment_metadata') ? wp_get_attachment_metadata($post_id, true) : get_post_meta($post_id, '_wp_attachment_metadata', true);
    if (is_array($meta) && !empty($meta['sizes']) && is_array($meta['sizes'])) {
        $dir = trailingslashit(dirname($relative_path));
        foreach ($meta['sizes'] as $size) {
            if (!empty($size['file'])) {
                $files[] = $dir . $size['file'];
            }
        }
    }

    $base = rtrim(remote_uploader_get_setting('ftp_basedir'), '/');
    $protocol = strtolower(remote_uploader_get_setting('protocol'));
    $host = remote_uploader_get_setting('ftp_host');
    $port = intval(remote_uploader_get_setting('port'));
    if ($port <= 0) {
        $port = ($protocol === 'sftp') ? 22 : 21;
    }

    if ($protocol === 'sftp') {
        if (!function_exists('ssh2_connect')) {
            error_log('❌ [Remote Uploader] Cannot delete via SFTP: ext-ssh2 missing');
            return;
        }
        $conn = @ssh2_connect($host, $port);
        if (!$conn) {
            error_log('❌ [Remote Uploader] Could not connect via SFTP for deletion');
            return;
        }
        $user = remote_uploader_get_setting('ftp_username');
        $pass = remote_uploader_get_setting('ftp_password');
        if (!@ssh2_auth_password($conn, $user, $pass)) {
            error_log('❌ [Remote Uploader] SFTP auth failed during deletion');
            return;
        }
        $sftp = @ssh2_sftp($conn);
        if (!$sftp) {
            error_log('❌ [Remote Uploader] Failed to initialize SFTP subsystem for deletion');
            return;
        }
        foreach ($files as $rel) {
            $remote = $base . '/' . ltrim($rel, '/');
            if (!@unlink('ssh2.sftp://' . $sftp . $remote)) {
                error_log('❌ [Remote Uploader] Failed to delete remote SFTP file: ' . $rel);
            } else {
                error_log('✅ [Remote Uploader] Deleted remote SFTP file: ' . $rel);
            }
        }
        return;
    }

    // FTP deletion
    $ftp_conn = @ftp_connect($host, $port);
    if (!$ftp_conn) {
        error_log('❌ [Remote Uploader] Could not connect to FTP for deletion');
        return;
    }
    $login = @ftp_login($ftp_conn, remote_uploader_get_setting('ftp_username'), remote_uploader_get_setting('ftp_password'));
    if (!$login) {
        error_log('❌ [Remote Uploader] FTP login failed during deletion');
        ftp_close($ftp_conn);
        return;
    }
    ftp_pasv($ftp_conn, true);

    foreach ($files as $rel) {
        $remote = $base . '/' . ltrim($rel, '/');
        if (!@ftp_delete($ftp_conn, $remote)) {
            error_log('❌ [Remote Uploader] Failed to delete remote FTP file: ' . $rel);
        } else {
            error_log('✅ [Remote Uploader] Deleted remote FTP file: ' . $rel);
        }
    }

    ftp_close($ftp_conn);
}

// Track offloaded files shortly so we can tag the new attachment record
function remote_uploader_mark_offloaded_by_path($relative_path) {
    $list = get_transient('remote_uploader_offloaded_files');
    if (!is_array($list)) $list = [];
    if (!in_array($relative_path, $list, true)) {
        $list[] = $relative_path;
        set_transient('remote_uploader_offloaded_files', $list, 10 * MINUTE_IN_SECONDS);
    }
}

add_action('add_attachment', 'remote_uploader_flag_offloaded_on_add');
function remote_uploader_flag_offloaded_on_add($post_id) {
    $relative = get_post_meta($post_id, '_wp_attached_file', true);
    if (!$relative) return;
    $list = get_transient('remote_uploader_offloaded_files');
    $is_offloaded = is_array($list) && in_array($relative, $list, true);
    if (!$is_offloaded) {
        $local = WP_CONTENT_DIR . '/uploads/' . ltrim($relative, '/');
        if (!file_exists($local) && remote_uploader_is_extension_allowed($relative)) {
            $is_offloaded = true;
        }
    }
    if ($is_offloaded) {
        update_post_meta($post_id, '_remote_uploader_offloaded', '1');
    }
}

add_filter('wp_prepare_attachment_for_js', 'remote_uploader_prepare_attachment_for_js', 10, 3);
function remote_uploader_prepare_attachment_for_js($response, $attachment, $meta) {
    $off = get_post_meta($attachment->ID, '_remote_uploader_offloaded', true) === '1';
    if (!$off) {
        $relative = get_post_meta($attachment->ID, '_wp_attached_file', true);
        if ($relative) {
            $local = WP_CONTENT_DIR . '/uploads/' . ltrim($relative, '/');
            if (!file_exists($local) && remote_uploader_is_extension_allowed($relative)) {
                $off = true;
            }
        }
    }
    $response['remoteUploaderOffloaded'] = (bool) $off;
    return $response;
}

add_action('admin_enqueue_scripts', 'remote_uploader_enqueue_admin_badge');
function remote_uploader_enqueue_admin_badge($hook) {
    if ($hook !== 'upload.php' && $hook !== 'media-new.php' && $hook !== 'post.php' && $hook !== 'post-new.php') {
        return;
    }
    // Ensure media scripts are present
    if (wp_script_is('media-grid', 'registered')) {
        wp_enqueue_script('media-grid');
    } else {
        wp_enqueue_script('media-views');
        if (wp_script_is('media', 'registered')) {
            wp_enqueue_script('media');
        }
    }
    // Minimal CSS badge + outline
    $css = '.remote-uploader-badge{position:absolute;top:4px;right:4px;background:#000;color:#fff;border-radius:9999px;font-size:10px;padding:2px 6px;line-height:1;z-index:100;opacity:.95;pointer-events:none} .attachments-browser .attachment, .media-modal .attachment{position:relative} .attachments-browser .attachment.remote-uploader-offloaded{box-shadow:0 0 0 3px #111 inset;border-radius:4px} .media-modal .attachment.remote-uploader-offloaded{box-shadow:0 0 0 3px #111 inset;border-radius:4px} .remote-uploader-health-banner{position:fixed;left:0;right:0;top:0;background:#8b0000;color:#fff;padding:10px 16px;font-weight:600;z-index:100000;text-align:center;display:none} .remote-uploader-health-banner a{color:#fff;text-decoration:underline} .remote-uploader-health-banner a:hover,.remote-uploader-health-banner a:focus{color:#fff;opacity:.95;text-decoration:underline}';
    wp_register_style('remote-uploader-admin-inline', false);
    wp_enqueue_style('remote-uploader-admin-inline');
    wp_add_inline_style('remote-uploader-admin-inline', $css);

    $cdn = esc_js(rtrim(remote_uploader_get_setting('subdomain_url'), '/'));
    $js = <<<'JS'
;(function(wp,$,CDN){if(!wp||!wp.media||!wp.media.view){return;}function mark($el){try{var img=$el.find('img');var src=img && img.attr && img.attr('src') || '';var off=$el.data('remoteUploaderOffloaded');if(!off&&CDN&&src && src.indexOf(CDN+'/')===0){off=true;}$el.toggleClass('remote-uploader-offloaded', !!off);$el.find('.remote-uploader-badge').remove();if(off){$el.append('<span class="remote-uploader-badge">CDN</span>');}}catch(e){}}
var Base=wp.media.view.Attachment;if(!Base){return;}var View=Base.extend({render:function(){Base.prototype.render.apply(this,arguments);var off=false;try{off=this.model&&this.model.get&&this.model.get('remoteUploaderOffloaded');}catch(e){}if(off){this.$el.data('remoteUploaderOffloaded',!!off);}mark(this.$el);return this;}});wp.media.view.Attachment=View;
// Fallback: mark existing elements after DOM ready
$(function(){setTimeout(function(){ $('.attachments-browser .attachment').each(function(){mark($(this));});},300);});})(window.wp, window.jQuery, '%CDN%');
JS;
    $js = str_replace('%CDN%', $cdn, $js);

    // Health banner polling
    $nonce = wp_create_nonce('remote_uploader');
    $health_js = <<<'JS'
;(function($,NONCE,CDN){function banner(){var $b=$('.remote-uploader-health-banner');if(!$b.length){$b=$('<div class="remote-uploader-health-banner"/>').hide();$('body').append($b);}return $b;}function setText($b){var url=CDN||'CDN';$b.html('CDN OFFLINE: '+url+' has been detected as offline (<a href="#" class="ru-try-again">Try Again</a>)');}function show(){var $b=banner();setText($b);var top=0;var $bar=$('#wpadminbar');if($bar.length){top=$bar.height();}$b.css({top:top+'px'}).stop(true,true).fadeIn(150);}function hide(){banner().stop(true,true).fadeOut(150);}function ping(){try{$.post(ajaxurl,{action:'remote_uploader_health',_wpnonce:NONCE}).done(function(r){if(r&&r.success){hide();}else{show();}}).fail(function(){show();});}catch(e){show();}}$(document).on('click','.remote-uploader-health-banner .ru-try-again',function(e){e.preventDefault();ping();});$(function(){ping();setInterval(ping,180000);});})(jQuery,'%NONCE%','%CDN%');
JS;
    $health_js = str_replace('%NONCE%', esc_js($nonce), $health_js);
    $health_js = str_replace('%CDN%', $cdn, $health_js);

    if (wp_script_is('media-grid', 'enqueued')) {
        wp_add_inline_script('media-grid', $js);
        wp_add_inline_script('media-grid', $health_js);
    }
    if (wp_script_is('media', 'enqueued')) {
        wp_add_inline_script('media', $js);
        wp_add_inline_script('media', $health_js);
    }
    if (wp_script_is('media-views', 'enqueued')) {
        wp_add_inline_script('media-views', $js);
        wp_add_inline_script('media-views', $health_js);
    }
}

// Build CDN health URL from configured subdomain_url
function remote_uploader_get_cdn_health_url() {
	$cdn = remote_uploader_get_setting('subdomain_url');
	if (!is_string($cdn) || $cdn === '') return '';
	$parts = wp_parse_url($cdn);
	if (empty($parts['scheme']) || empty($parts['host'])) return '';
	$port = isset($parts['port']) ? (':' . intval($parts['port'])) : '';
	return $parts['scheme'] . '://' . $parts['host'] . $port . '/_health';
}

// AJAX: Health check
add_action('wp_ajax_remote_uploader_health', 'remote_uploader_ajax_health');
function remote_uploader_ajax_health() {
	check_ajax_referer('remote_uploader');
	if (!current_user_can('upload_files')) {
		wp_send_json_error([ 'message' => 'Permission denied' ], 403);
	}
	$url = remote_uploader_get_cdn_health_url();
	if (!$url) {
		wp_send_json_error([ 'message' => 'CDN Offline' ]);
	}
	$resp = wp_remote_get($url, [ 'timeout' => 3 ]);
	if (is_wp_error($resp)) {
		wp_send_json_error([ 'message' => 'CDN Offline' ]);
	}
	$code = intval(wp_remote_retrieve_response_code($resp));
	$body = trim((string) wp_remote_retrieve_body($resp));
	if ($code === 200 && strtolower($body) === 'ok') {
		wp_send_json_success();
	}
	wp_send_json_error([ 'message' => 'CDN Offline' ]);
}
