<?php
/**
 * FLW Starter Updater (single-file)
 *
 * Minimal self-hosted updater for WordPress plugins without API key checks.
 *
 * Usage in your plugin main file:
 *
 * require_once __DIR__ . '/starter.php';
 * FLW_Starter_Update::boot(__FILE__, 'your-plugin-slug', 'https://your-domain.com/updates/');
 *
 * The updater expects your endpoint to return a JSON metadata document similar to Plugin Update Checker:
 *   - version (string, required)
 *   - download_url (string, required)
 *   - tested (string, optional)
 *   - requires (string, optional)
 *   - requires_php (string, optional)
 *   - last_updated (string, optional)
 *   - homepage (string, optional)
 *   - sections (object, optional) e.g. {"description": "...", "changelog": "..."}
 *   - banners (object, optional) e.g. {"low": "https://...", "high": "https://..."}
 *
 * The metadata will be fetched at: {metadataBase}?action=get_metadata&slug={pluginSlug}
 */

// Prevent direct access
if ( ! defined('ABSPATH') ) {
    exit;
}

if ( ! class_exists('FLW_Starter_Update') ) {
    class FLW_Starter_Update {
        /** @var string */
        private $pluginFile;
        /** @var string */
        private $pluginBasename;
        /** @var string */
        private $pluginSlug;
        /** @var string */
        private $metadataBase;

        /**
         * Boot the updater for a specific plugin.
         *
         * @param string $pluginFile   Path to the plugin main file, usually __FILE__ from caller
         * @param string $pluginSlug   Unique plugin slug (defaults to directory name of the plugin)
         * @param string $metadataBase Base URL to your update metadata endpoint (trailing slash OK)
         */
        public static function boot( $pluginFile, $pluginSlug = null, $metadataBase = null ) {
            static $instances = [];
            $key = md5( $pluginFile );
            if ( isset( $instances[$key] ) ) {
                return $instances[$key];
            }
            $instances[$key] = new self( $pluginFile, $pluginSlug, $metadataBase );
            return $instances[$key];
        }

        private function __construct( $pluginFile, $pluginSlug, $metadataBase ) {
            $this->pluginFile     = $pluginFile;
            $this->pluginBasename = plugin_basename( $pluginFile );
            $this->pluginSlug     = $pluginSlug ?: dirname( $this->pluginBasename );
            $this->metadataBase   = $this->determineMetadataBase( $metadataBase );

            // Hook into WordPress update system
            add_filter( 'site_transient_update_plugins', [ $this, 'injectUpdate' ] );
            add_filter( 'plugins_api', [ $this, 'providePluginInfo' ], 10, 3 );

            // Optional action link to force a check
            add_filter( 'plugin_action_links_' . $this->pluginBasename, [ $this, 'addActionLinks' ] );
            add_action( 'admin_post_flw_starter_force_update', [ $this, 'handleForceUpdate' ] );
        }

        private function determineMetadataBase( $metadataBase ) {
            // Priority: passed param → constant → filter → default
            if ( is_string( $metadataBase ) && $metadataBase !== '' ) {
                return $this->normalizeBase( $metadataBase );
            }
            if ( defined( 'FLW_STARTER_METADATA_URL' ) && FLW_STARTER_METADATA_URL ) {
                return $this->normalizeBase( FLW_STARTER_METADATA_URL );
            }
            $filtered = apply_filters( 'flw_starter_metadata_url', '' );
            if ( is_string( $filtered ) && $filtered !== '' ) {
                return $this->normalizeBase( $filtered );
            }
            // Fallback default; replace with your domain if desired
            return $this->normalizeBase( 'https://frostlineworks.com/updates/' );
        }

        private function normalizeBase( $base ) {
            // Ensure it ends with a slash or query continuation is valid when using add_query_arg
            return rtrim( $base, "?/\t\n\r\0\x0B" ) . '/';
        }

        /**
         * Inject update object into the update transient when a newer version is available.
         *
         * @param object $transient
         * @return object
         */
        public function injectUpdate( $transient ) {
            if ( empty( $transient ) || ! is_object( $transient ) ) {
                return $transient;
            }

            $meta = $this->fetchMetadata();
            if ( ! $meta || empty( $meta['version'] ) || empty( $meta['download_url'] ) ) {
                return $transient;
            }

            $currentVersion = $this->getInstalledVersion();
            if ( ! $currentVersion || version_compare( $meta['version'], $currentVersion, '>' ) ) {
                $update              = new stdClass();
                $update->slug        = $this->pluginSlug;
                $update->plugin      = $this->pluginBasename;
                $update->new_version = $meta['version'];
                $update->package     = $meta['download_url'];
                $update->url         = isset( $meta['homepage'] ) ? $meta['homepage'] : '';

                if ( ! empty( $meta['tested'] ) ) {
                    $update->tested = $meta['tested'];
                }
                if ( ! empty( $meta['requires'] ) ) {
                    $update->requires = $meta['requires'];
                }
                if ( ! empty( $meta['icons'] ) && is_array( $meta['icons'] ) ) {
                    $update->icons = $meta['icons'];
                }

                if ( ! isset( $transient->response ) || ! is_array( $transient->response ) ) {
                    $transient->response = [];
                }
                $transient->response[ $this->pluginBasename ] = $update;
            }

            return $transient;
        }

        /**
         * Provide detailed plugin info for the "View details" modal in the Plugins screen.
         *
         * @param false|object $result
         * @param string       $action
         * @param object       $args
         * @return object|false
         */
        public function providePluginInfo( $result, $action, $args ) {
            if ( $action !== 'plugin_information' || empty( $args->slug ) || $args->slug !== $this->pluginSlug ) {
                return $result;
            }

            $meta = $this->fetchMetadata();
            if ( ! $meta || empty( $meta['version'] ) ) {
                return $result;
            }

            $info = new stdClass();
            $info->name          = $this->getPluginHeaderValue( 'Name' ) ?: ucfirst( str_replace( '-', ' ', $this->pluginSlug ) );
            $info->slug          = $this->pluginSlug;
            $info->version       = $meta['version'];
            $info->requires      = isset( $meta['requires'] ) ? $meta['requires'] : '';
            $info->tested        = isset( $meta['tested'] ) ? $meta['tested'] : '';
            $info->requires_php  = isset( $meta['requires_php'] ) ? $meta['requires_php'] : '';
            $info->last_updated  = isset( $meta['last_updated'] ) ? $meta['last_updated'] : '';
            $info->homepage      = isset( $meta['homepage'] ) ? $meta['homepage'] : '';
            $info->author        = $this->getPluginHeaderValue( 'Author' ) ?: '';
            $info->author_profile= $this->getPluginHeaderValue( 'AuthorURI' ) ?: '';
            $info->sections      = isset( $meta['sections'] ) && is_array( $meta['sections'] ) ? $meta['sections'] : [ 'description' => $this->getPluginHeaderValue( 'Description' ) ?: '' ];
            if ( ! empty( $meta['banners'] ) && is_array( $meta['banners'] ) ) {
                $info->banners = $meta['banners'];
            }
            if ( ! empty( $meta['download_url'] ) ) {
                $info->download_link = $meta['download_url'];
            }
            return $info;
        }

        /** Add action link to force update check */
        public function addActionLinks( $links ) {
            $url   = wp_nonce_url( admin_url( 'admin-post.php?action=flw_starter_force_update' ), 'flw_starter_force_update' );
            $links[] = '<a href="' . esc_url( $url ) . '">' . esc_html__( 'Check for updates', 'flw-starter' ) . '</a>';
            return $links;
        }

        /** Handle the force update admin-post action */
        public function handleForceUpdate() {
            if ( ! current_user_can( 'update_plugins' ) ) {
                wp_die( 'Unauthorized user' );
            }
            check_admin_referer( 'flw_starter_force_update' );

            if ( function_exists( 'wp_clean_plugins_cache' ) ) {
                wp_clean_plugins_cache( true );
            } else {
                delete_site_transient( 'update_plugins' );
            }
            if ( function_exists( 'wp_update_plugins' ) ) {
                wp_update_plugins();
            }
            wp_safe_redirect( add_query_arg( 'flw_updates_checked', '1', admin_url( 'plugins.php' ) ) );
            exit;
        }

        /** Get current installed version of the plugin */
        private function getInstalledVersion() {
            if ( ! function_exists( 'get_plugin_data' ) ) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }
            $data = get_plugin_data( $this->pluginFile, false, false );
            return isset( $data['Version'] ) ? $data['Version'] : '';
        }

        /** Fetch update metadata from self-hosted endpoint */
        private function fetchMetadata() {
            $url = add_query_arg(
                [
                    'action' => 'get_metadata',
                    'slug'   => $this->pluginSlug,
                ],
                $this->metadataBase
            );

            $response = wp_remote_get( $url, [ 'timeout' => 15 ] );
            if ( is_wp_error( $response ) ) {
                return null;
            }
            $code = wp_remote_retrieve_response_code( $response );
            if ( $code !== 200 ) {
                return null;
            }
            $body = wp_remote_retrieve_body( $response );
            $data = json_decode( $body, true );
            if ( ! is_array( $data ) ) {
                return null;
            }
            return $data;
        }

        /** Read a field from the plugin header */
        private function getPluginHeaderValue( $field ) {
            if ( ! function_exists( 'get_plugin_data' ) ) {
                require_once ABSPATH . 'wp-admin/includes/plugin.php';
            }
            $data = get_plugin_data( $this->pluginFile, false, false );
            return isset( $data[ $field ] ) ? $data[ $field ] : '';
        }
    }
}

// No automatic boot here to keep this file reusable across multiple plugins.
// In your plugin, call: FLW_Starter_Update::boot(__FILE__, 'your-plugin-slug', 'https://your-domain.com/updates/');


